/*
 * Copyright 2022 Nedra Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package digital.nedra.commons.starter.keycloak.session.config.support;

import java.io.IOException;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpServletResponseWrapper;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;
import org.springframework.web.filter.OncePerRequestFilter;

@RequiredArgsConstructor
@Component
@Order(Ordered.HIGHEST_PRECEDENCE)
public class XhrRedirectFilter extends OncePerRequestFilter {

  public static final String XML_HTTP_REQUEST = "XMLHttpRequest";
  public static final String X_REQUESTED_WITH = "X-Requested-With";
  public static final String ACCESS_DENIED = "Access Denied";
  @Value("${starters.keycloak.redirect-mode}")
  private String redirectMode;
  @Value("${starters.keycloak.url-holder-header}")
  private String urlHolderHeader;

  @Override
  protected void doFilterInternal(HttpServletRequest request,
                                  HttpServletResponse response, FilterChain filterChain)
      throws ServletException, IOException {
    filterChain.doFilter(request, new HttpServletResponseWrapper(response) {
      @Override
      public void sendRedirect(String location) throws IOException {
        if (RedirectMode.ALWAYS_REDIRECT.name().equals(redirectMode)) {
          super.sendRedirect(location);
        } else if (RedirectMode.HEADER_BASED.name().equals(redirectMode)) {
          if (XML_HTTP_REQUEST.equals(request.getHeader(X_REQUESTED_WITH))) {
            sendXhrRedirect(location);
          } else {
            super.sendRedirect(location);
          }
        } else {
          throw new RuntimeException("Unsupported redirect mode: " + redirectMode);
        }
      }


      private void sendXhrRedirect(String location) throws IOException {
        super.setHeader(urlHolderHeader, location);
        super.sendError(HttpServletResponse.SC_UNAUTHORIZED, ACCESS_DENIED);
      }
    });
  }

  public enum RedirectMode {
    ALWAYS_REDIRECT, HEADER_BASED
  }
}
