/*
 * Copyright 2022 Nedra Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package digital.nedra.commons.starter.keycloak.session.config.support;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.ResponseEntity;
import org.springframework.security.core.Authentication;
import org.springframework.security.oauth2.core.oidc.user.OidcUser;
import org.springframework.security.web.authentication.logout.SecurityContextLogoutHandler;
import org.springframework.stereotype.Component;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

@Slf4j
@RequiredArgsConstructor
@Component
public class KeycloakLogoutHandler extends SecurityContextLogoutHandler {

  private final RestTemplate restTemplate = new RestTemplate();
  @Value("${starters.keycloak.protocol-url}")
  private String protocolUrl;

  @Override
  public void logout(HttpServletRequest request, HttpServletResponse response,
                     Authentication authentication) {
    super.logout(request, response, authentication);
    if (authentication == null) {
      log.info("The user has already logged out");
      return;
    }

    OidcUser user = (OidcUser) authentication.getPrincipal();

    String endSessionEndpoint = protocolUrl + "/logout";

    UriComponentsBuilder builder = UriComponentsBuilder //
        .fromUriString(endSessionEndpoint) //
        .queryParam("id_token_hint", user.getIdToken().getTokenValue());

    ResponseEntity<String> logoutResponse =
        restTemplate.getForEntity(builder.toUriString(), String.class);
    if (logoutResponse.getStatusCode().is2xxSuccessful()) {
      log.info("Successfulley logged out in Keycloak");
    } else {
      log.info("Could not propagate logout to Keycloak");
    }
  }

}