/*
 * Copyright 2022 Nedra Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package digital.nedra.commons.starter.keycloak.session.config.support;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import javax.annotation.PostConstruct;
import lombok.NoArgsConstructor;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.AuthorityUtils;
import org.springframework.security.core.authority.mapping.SimpleAuthorityMapper;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

@NoArgsConstructor
@Component
public class KeycloakAuthoritiesExtractor {
  public static final String ROLES = "roles";
  private final SimpleAuthorityMapper authoritiesMapper = new SimpleAuthorityMapper();

  @PostConstruct
  public void init() {
    authoritiesMapper.setConvertToUpperCase(true);
  }

  @SuppressWarnings("unchecked")
  public List<GrantedAuthority> extractRealmAuthorities(Map<String, Object> resourceMap) {
    List<String> authorities = (List<String>) resourceMap.get(ROLES);
    if (CollectionUtils.isEmpty(authorities)) {
      return Collections.emptyList();
    }

    return map(AuthorityUtils.createAuthorityList(authorities.toArray(String[]::new)));
  }

  @SuppressWarnings("unchecked")
  public List<GrantedAuthority> extractClientAuthorities(
      String clientId, Map<String, Object> resourceMap) {

    Map<String, Map<String, Object>> clientResource =
        (Map<String, Map<String, Object>>) resourceMap.get(clientId);

    if (CollectionUtils.isEmpty(clientResource)) {
      return Collections.emptyList();
    }

    List<String> clientRoles = (List<String>) clientResource.get(ROLES);

    if (CollectionUtils.isEmpty(clientRoles)) {
      return Collections.emptyList();
    }

    return map(AuthorityUtils.createAuthorityList(clientRoles.toArray(String[]::new)));
  }

  /**
   * Maps given {@code authorities} using {@link #authoritiesMapper} if available.
   */
  private List<GrantedAuthority> map(List<GrantedAuthority> authorities) {
    return new ArrayList<>(authoritiesMapper.mapAuthorities(authorities));
  }
}
