/*
 * Copyright 2022 Nedra Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package digital.nedra.commons.starter.keycloak.session.config;

import digital.nedra.commons.starter.common.config.properties.SecurityProperties;
import digital.nedra.commons.starter.keycloak.session.config.support.KeycloakLogoutHandler;
import digital.nedra.commons.starter.keycloak.session.config.support.KeycloakOauth2UserService;
import digital.nedra.commons.starter.keycloak.session.config.support.XhrCookieRequestCache;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.config.Customizer;
import org.springframework.security.config.annotation.method.configuration.EnableGlobalMethodSecurity;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.builders.WebSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configuration.WebSecurityConfigurerAdapter;
import org.springframework.security.config.annotation.web.configurers.ExpressionUrlAuthorizationConfigurer;

@SuppressWarnings({"CheckStyle", "SpringJavaInjectionPointsAutowiringInspection"})
@EnableGlobalMethodSecurity(prePostEnabled = true)
@Configuration
@RequiredArgsConstructor
@EnableWebSecurity
public class SecurityConfigurerAdapter extends WebSecurityConfigurerAdapter {
  private final KeycloakLogoutHandler logoutHandler;
  private final XhrCookieRequestCache requestCache;
  private final KeycloakOauth2UserService keycloakOidcUserService;
  private final Customizer<ExpressionUrlAuthorizationConfigurer<HttpSecurity>
      .ExpressionInterceptUrlRegistry> authorizeRequestsCustomizer;
  private final SecurityProperties securityProperties;
  @Value("${spring.security.oauth2.client.registration.sso.provider}")
  private String oauth2Provider;

  @Override
  protected void configure(HttpSecurity http) throws Exception {
    http
        .authorizeRequests(authorizeRequestsCustomizer)
        .oauth2Login(httpSec -> httpSec
            .loginPage("/oauth2/authorization/" + oauth2Provider)
            .userInfoEndpoint()
            .oidcUserService(keycloakOidcUserService))
        .oauth2Client()
        .and()
        .logout()
        .addLogoutHandler(logoutHandler)
        .and()
        .csrf().disable()
        .cors()
        .and()
        .requestCache(i -> i.requestCache(requestCache));
  }

  //Set up ignored urls
  @Override
  public void configure(WebSecurity web) {
    final WebSecurity.IgnoredRequestConfigurer ignoredRequestConfigurer = web.ignoring();
    securityProperties.getIgnoredUrls().forEach(ignoredRequestConfigurer::antMatchers);
  }
}
