/*
 * Copyright 2022 Nedra Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package digital.nedra.commons.starter.keycloak.redis.service;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.security.core.Authentication;
import org.springframework.security.oauth2.client.OAuth2AuthorizedClient;
import org.springframework.security.oauth2.client.OAuth2AuthorizedClientId;
import org.springframework.security.oauth2.client.OAuth2AuthorizedClientService;
import org.springframework.security.oauth2.client.registration.ClientRegistration;
import org.springframework.security.oauth2.client.registration.ClientRegistrationRepository;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

@Component
@Slf4j
public class RedisOAuth2AuthorizedClientService implements OAuth2AuthorizedClientService {
  private final RedisTemplate<Object, Object> authorizedClients;
  private final ClientRegistrationRepository clientRegistrationRepository;

  @Autowired
  public RedisOAuth2AuthorizedClientService(
      @Qualifier("redisTemplate") RedisTemplate<Object, Object> authorizedClients,
      ClientRegistrationRepository clientRegistrationRepository) {
    this.authorizedClients = authorizedClients;
    this.clientRegistrationRepository = clientRegistrationRepository;
  }

  @Override
  @SuppressWarnings("unchecked")
  public <T extends OAuth2AuthorizedClient> T loadAuthorizedClient(
      String clientRegistrationId, String principalName) {
    Assert.hasText(clientRegistrationId, "clientRegistrationId cannot be empty");
    Assert.hasText(principalName, "principalName cannot be empty");
    ClientRegistration registration =
        clientRegistrationRepository.findByRegistrationId(clientRegistrationId);
    if (registration == null) {
      return null;
    }
    OAuth2AuthorizedClientId oauth2AuthorizedClientId =
        new OAuth2AuthorizedClientId(clientRegistrationId, principalName);

    return (T) this.authorizedClients.opsForHash()
     .get(oauth2AuthorizedClientId, oauth2AuthorizedClientId.hashCode());
  }

  @Override
  public void saveAuthorizedClient(OAuth2AuthorizedClient authorizedClient,
                                   Authentication principal) {
    Assert.notNull(authorizedClient, "authorizedClient cannot be null");
    Assert.notNull(principal, "principal cannot be null");

    OAuth2AuthorizedClientId oauth2AuthorizedClientId =
        new OAuth2AuthorizedClientId(
            authorizedClient.getClientRegistration().getRegistrationId(),
            principal.getName());
    authorizedClients.opsForHash()
        .put(oauth2AuthorizedClientId, oauth2AuthorizedClientId.hashCode(), authorizedClient);
  }

  @Override
  public void removeAuthorizedClient(String clientRegistrationId, String principalName) {
    Assert.hasText(clientRegistrationId, "clientRegistrationId cannot be empty");
    Assert.hasText(principalName, "principalName cannot be empty");
    ClientRegistration registration =
        clientRegistrationRepository.findByRegistrationId(clientRegistrationId);
    if (registration != null) {
      OAuth2AuthorizedClientId oauth2AuthorizedClientId =
          new OAuth2AuthorizedClientId(clientRegistrationId, principalName);
      authorizedClients.opsForHash()
          .delete(oauth2AuthorizedClientId, oauth2AuthorizedClientId.hashCode());
    }
  }
}
