/*
 * Copyright 2022 Nedra Team
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package digital.nedra.commons.starter.audit.service;

import digital.nedra.commons.starter.audit.config.AuditProperties;
import digital.nedra.commons.starter.audit.dto.AuditEvent;
import digital.nedra.commons.starter.audit.dto.Extension;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.StringJoiner;
import lombok.RequiredArgsConstructor;
import org.apache.commons.text.StringSubstitutor;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;

@Service
@RequiredArgsConstructor
public class AuditService {
  private final AuditPropertiesResolver auditPropertiesResolver;
  private final AuditProperties auditProperties;
  private final AuditLogger auditLogger;

  public void logEvent(@NonNull AuditEvent event) {
    this.logEvent(event, null);
  }

  public void logEvent(@NonNull AuditEvent event, Extension extension) {
    var values = new HashMap<String, Object>();

    fillFromProps(values);
    fillFromEvent(event, values);

    values.put("extension", Optional.ofNullable(extension)
        .map(this::buildExtension)
        .orElse(""));

    var substitutor = new StringSubstitutor(values);
    String result = substitutor.replace(auditProperties.getCef().getTemplate());
    auditLogger.logEvent(result);
  }

  private String buildExtension(Extension ext) {
    var stringJoiner = new StringJoiner(" ", "", "|");

    addClearedValueIfPresent(stringJoiner, "src", ext.src());
    addClearedValueIfPresent(stringJoiner, "dst", ext.dst());
    addClearedValueIfPresent(stringJoiner, "shost", ext.shost());
    addClearedValueIfPresent(stringJoiner, "suid", ext.suid());
    addClearedValueIfPresent(stringJoiner, "suser", ext.suser());
    addClearedValueIfPresent(stringJoiner, "msg", ext.msg());

    Optional.ofNullable(ext.end())
        .ifPresent(s -> stringJoiner.add("end=" + buildEndTime(s)));

    return stringJoiner.toString();
  }

  private void addClearedValueIfPresent(StringJoiner stringJoiner, String key, String value) {
    Optional.ofNullable(value)
        .map(this::cleanUp)
        .ifPresent(s -> stringJoiner.add(key + "=" + s));
  }

  private String buildEndTime(Long endTimeMillis) {
    AuditProperties.EndTime endTimeProperties = auditProperties.getEndTime();

    ZoneId zoneId = ZoneId.of(endTimeProperties.getTimeZone());
    ZonedDateTime endDateTime =
        ZonedDateTime.ofInstant(Instant.ofEpochMilli(endTimeMillis), zoneId);

    return Optional.ofNullable(endTimeProperties.getFormat())
        .map(DateTimeFormatter::ofPattern)
        .map(endDateTime::format)
        .orElseGet(() -> Long.toString(endDateTime.toInstant().toEpochMilli()));
  }

  private String cleanUp(String value) {
    return Optional.ofNullable(value)
        .map(i -> i.replace("=", ""))
        .map(i -> i.replace("|", ""))
        .map(i -> i.replaceAll("[\\t\\n\\r]+", " "))
        .orElse(null);
  }

  private void fillFromEvent(AuditEvent event, Map<String, Object> values) {
    values.put("event.id", cleanUp(event.id()));
    values.put("event.name", cleanUp(event.name()));
    values.put("event.severity", event.severity().getValue());
  }

  private void fillFromProps(Map<String, Object> values) {
    values.put("cef.version", auditProperties.getCef().getVersion());
    values.put("company.name", cleanUp(auditProperties.getCompany().getName()));
    values.put("product.name", cleanUp(auditPropertiesResolver.getProductName().orElse("")));
    values.put("product.version", auditPropertiesResolver.getProductVersion().orElse(""));
  }
}
