/*
 * Decompiled with CFR 0.152.
 */
package de.gematik.test.tiger.testenvmgr.servers;

import de.gematik.test.tiger.common.data.config.CfgHelmChartOptions;
import de.gematik.test.tiger.testenvmgr.servers.AbstractTigerServer;
import de.gematik.test.tiger.testenvmgr.servers.TigerServerStatus;
import de.gematik.test.tiger.testenvmgr.servers.log.TigerStreamLogFeeder;
import de.gematik.test.tiger.testenvmgr.util.TigerTestEnvException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.event.Level;

public class KubeUtils {
    public static final String PORT_EXCEPTION_MESSAGE = "Failed to start kubectl port forward for command ";
    private final String helmCommand;
    private final String kubeCtlCommand;
    private final AbstractTigerServer tigerServer;
    private String workingDirectory;
    private final Logger log;
    private final Executor executor;
    private final CopyOnWriteArrayList<Process> processes = new CopyOnWriteArrayList();
    private final Map<String, TigerServerStatus> startupPhaseStatus = new HashMap<String, TigerServerStatus>();

    public KubeUtils(AbstractTigerServer server, Executor executor) {
        String executableExtension = System.getProperty("os.name").startsWith("Win") ? ".exe" : "";
        this.helmCommand = server.findCommandInPath("helm" + executableExtension);
        this.kubeCtlCommand = server.findCommandInPath("kubectl" + executableExtension);
        this.tigerServer = server;
        this.executor = executor;
        this.log = this.tigerServer.getLog();
    }

    public void setKubernetesContext(String context) {
        if (context == null) {
            return;
        }
        this.log.info("Setting kubernetes context for helm chart {} to {}...", (Object)this.tigerServer.getServerId(), (Object)context);
        ProcessBuilder processBuilder = new ProcessBuilder(new String[0]).command(this.kubeCtlCommand, "config", "use-context", context).redirectErrorStream(true);
        this.applyEnvPropertiesToProcess(processBuilder);
        int exitCode = this.getSafely(this.spinUpNewExternalProcess(processBuilder), "set context " + context + " for server " + this.tigerServer.getServerId()).exitValue();
        if (exitCode != 0) {
            throw new TigerTestEnvException("Setting context '%s' yielded an error exit code from kubectl for server %s!", new Object[]{context, this.tigerServer.getServerId()});
        }
    }

    public CompletableFuture<Process> startupHelmChart() {
        ProcessBuilder processBuilder = new ProcessBuilder(new String[0]).command(ListUtils.union(List.of(this.helmCommand), this.buildStartupCommandOptions())).directory(new File(this.workingDirectory)).redirectErrorStream(true).inheritIO();
        this.applyEnvPropertiesToProcess(processBuilder);
        return this.spinUpNewExternalProcess(processBuilder);
    }

    private List<String> buildStartupCommandOptions() {
        ArrayList<String> command = new ArrayList<String>();
        command.add("upgrade");
        command.add("--install");
        CfgHelmChartOptions options = this.tigerServer.getConfiguration().getHelmChartOptions();
        if (options.isDebug()) {
            command.add("--debug");
        }
        if (this.tigerServer.getConfiguration().getVersion() != null) {
            command.add("--version");
            command.add(this.tigerServer.getConfiguration().getVersion());
        }
        if (options.getNameSpace() != null) {
            command.add("--namespace");
            command.add(options.getNameSpace());
        }
        if (options.getValues() != null) {
            options.getValues().forEach(value -> {
                command.add("--set");
                command.add((String)value);
            });
        }
        command.add("--timeout");
        command.add(this.tigerServer.getConfiguration().getStartupTimeoutSec() + "s");
        command.add(options.getPodName());
        command.add((String)this.tigerServer.getConfiguration().getSource().get(0));
        return command;
    }

    public void exposePortsViaKubectl(CfgHelmChartOptions options) {
        List exposedPorts = options.getExposedPorts();
        List<String> serviceNames = this.getKubernetesServices();
        exposedPorts.parallelStream().map(entry -> entry.replaceAll("\\s", "")).map(colonSeparatedValues -> Arrays.stream(colonSeparatedValues.split(",")).toList()).forEach(values -> {
            Optional<String> serviceName = serviceNames.stream().filter(svcName -> svcName.equals(values.get(0)) || svcName.matches((String)values.get(0))).findAny();
            if (serviceName.isPresent()) {
                String podName = serviceName.get();
                this.log.info("Exposing ports {} of service {} for helm chart {}...", new Object[]{values.subList(1, values.size()), podName, this.tigerServer.getServerId()});
                ArrayList<String> cmd = new ArrayList<String>();
                cmd.add(this.kubeCtlCommand);
                cmd.add("--namespace");
                cmd.add(KubeUtils.getNameSpaceOrDefault(options));
                cmd.add("port-forward");
                cmd.add("service/" + podName);
                cmd.addAll(values.subList(1, values.size()));
                ProcessBuilder processBuilder = new ProcessBuilder(new String[0]).command(cmd).inheritIO().redirectErrorStream(true);
                this.checkKubeCtlPortForwarding(processBuilder, StringUtils.join(cmd, (String)" "));
            }
        });
    }

    private static String getNameSpaceOrDefault(CfgHelmChartOptions options) {
        return KubeUtils.getNameSpaceOrDefault(options.getNameSpace());
    }

    private static String getNameSpaceOrDefault(String nameSpace) {
        return nameSpace != null ? nameSpace : "default";
    }

    private List<String> getKubernetesStatusLines(String nameSpace) {
        ProcessBuilder processBuilder = new ProcessBuilder(new String[0]).command(this.kubeCtlCommand, "get", "pods", "-o", "wide", "-n", KubeUtils.getNameSpaceOrDefault(nameSpace)).redirectErrorStream(true);
        return (List)this.getSafely((CompletableFuture)this.spinUpNewExternalProcess(processBuilder).thenApplyAsync(process -> {
            InputStream input = process.getInputStream();
            try {
                return Arrays.stream(IOUtils.toString((InputStream)input, (Charset)StandardCharsets.UTF_8).split("\n")).skip(1L).toList();
            }
            catch (IOException e) {
                throw new TigerTestEnvException("Unable to retrieve list of pods from kubernetes cluster!", (Throwable)e);
            }
        }), "get list of pods for server " + this.tigerServer.getServerId());
    }

    private List<String> getKubernetesServices() {
        String nameSpace = KubeUtils.getNameSpaceOrDefault(this.tigerServer.getConfiguration().getHelmChartOptions());
        ProcessBuilder processBuilder = new ProcessBuilder(new String[0]).command(this.kubeCtlCommand, "get", "services", "-n", nameSpace).redirectErrorStream(true);
        return (List)this.getSafely((CompletableFuture)this.spinUpNewExternalProcess(processBuilder).thenApplyAsync(process -> {
            InputStream input = process.getInputStream();
            try {
                return Arrays.stream(IOUtils.toString((InputStream)input, (Charset)StandardCharsets.UTF_8).split("\n")).skip(1L).map(line -> line.substring(0, line.indexOf(" "))).toList();
            }
            catch (IOException e) {
                throw new TigerTestEnvException("Unable to retrieve list of services from kubernetes cluster!", (Throwable)e);
            }
        }), "get list of services in cluster for server " + this.tigerServer.getServerId());
    }

    private void checkKubeCtlPortForwarding(ProcessBuilder processBuilder, String command) {
        boolean hasExited;
        try {
            hasExited = this.getSafely(this.spinUpNewExternalProcess(processBuilder), "start kubectl port forward").waitFor(1L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            this.log.error("Failed to start kubectl port forward - InterruptedException {}", (Object)command);
            Thread.currentThread().interrupt();
            throw new TigerTestEnvException("Failed to start kubectl port forward - InterruptedException {}", (Throwable)e);
        }
        if (hasExited) {
            this.log.error("Failed to start kubectl port forward for command  {}", (Object)command);
            throw new TigerTestEnvException("Failed to start kubectl port forward for command \"%s\"! Please check exposedPorts in your tiger.yaml!", new Object[]{command});
        }
    }

    private Optional<String> getStatusLineForPod(List<String> statusLines, String podName) {
        return statusLines.stream().filter(line -> {
            String podnm = line.split(" +")[0];
            return podnm.matches(podName) || podnm.equals(podName);
        }).findFirst();
    }

    public long getNumOfPodsOnStatusList(String nameSpace) {
        return this.getKubernetesStatusLines(nameSpace).stream().map(line -> line.split(" +")).filter(columns -> this.tigerServer.getConfiguration().getHelmChartOptions().getHealthcheckPods().stream().anyMatch(podName -> columns[0].equals(podName) || columns[0].matches((String)podName))).count();
    }

    public long getNumOfRunningPods(String nameSpace) {
        List<String> statusLines = this.getKubernetesStatusLines(nameSpace);
        return this.tigerServer.getConfiguration().getHelmChartOptions().getHealthcheckPods().stream().filter(podName -> this.isPodRunning((String)podName, statusLines)).count();
    }

    private boolean isPodRunning(String podName, List<String> statusLines) {
        return statusLines.stream().map(line -> line.split(" +")).filter(columns -> columns[0].equals(podName) || columns[0].matches(podName)).anyMatch(columns -> {
            TigerServerStatus newStatus = KubeUtils.getTigerServerStatusFromKubeCtlStatus(columns);
            if (this.startupPhaseStatus.getOrDefault(podName, TigerServerStatus.NEW) != newStatus) {
                this.startupPhaseStatus.put(podName, newStatus);
                if (newStatus == TigerServerStatus.STOPPED) {
                    this.log.warn("Status of pod {} STOPPED ({}) unexpectedly", (Object)columns[0], (Object)columns[2]);
                } else if (this.tigerServer.getConfiguration().getHelmChartOptions().isDebug()) {
                    this.log.info("Status of pod {} switched to {}", (Object)columns[0], (Object)newStatus);
                }
            }
            return newStatus == TigerServerStatus.RUNNING;
        });
    }

    private static TigerServerStatus getTigerServerStatusFromKubeCtlStatus(String[] columns) {
        return switch (columns[2]) {
            case "ContainerCreating", "Pending" -> TigerServerStatus.STARTING;
            case "Running" -> {
                String[] ready = columns[1].split("/");
                if (ready.length == 2 && ready[0].equals(ready[1])) {
                    yield TigerServerStatus.RUNNING;
                }
                yield TigerServerStatus.STARTING;
            }
            default -> TigerServerStatus.STOPPED;
        };
    }

    public Optional<CompletableFuture<Process>> shutdownHelm(String nameSpace) {
        ProcessBuilder processBuilder = new ProcessBuilder(new String[0]).command(this.helmCommand, "list", "-n", KubeUtils.getNameSpaceOrDefault(nameSpace));
        this.applyEnvPropertiesToProcess(processBuilder);
        CompletableFuture<Process> shutdownFuture = this.spinUpNewExternalProcess(processBuilder);
        try {
            String list = IOUtils.toString((InputStream)this.getSafely(shutdownFuture, "list helm charts").getInputStream(), (Charset)StandardCharsets.UTF_8);
            if (Arrays.stream(list.split("\n")).noneMatch(chartName -> chartName.startsWith(this.tigerServer.getConfiguration().getHelmChartOptions().getPodName()))) {
                this.log.warn("Helm chart {} not listed, no need to issue shutdown command!", (Object)this.tigerServer.getConfiguration().getHelmChartOptions().getPodName());
                return Optional.empty();
            }
        }
        catch (IOException e) {
            throw new TigerTestEnvException("Unable to obtain list of helm charts! Trying to shutdown nevertheless.", (Throwable)e);
        }
        processBuilder = new ProcessBuilder(new String[0]).command(this.helmCommand, "uninstall", "-n", KubeUtils.getNameSpaceOrDefault(nameSpace), "--wait", this.tigerServer.getConfiguration().getHelmChartOptions().getPodName());
        this.applyEnvPropertiesToProcess(processBuilder);
        return Optional.of(this.spinUpNewExternalProcess(processBuilder));
    }

    private CompletableFuture<Process> spinUpNewExternalProcess(ProcessBuilder processBuilder) {
        return CompletableFuture.supplyAsync(() -> {
            try {
                this.log.debug("Starting process {}", processBuilder.command());
                return processBuilder.start();
            }
            catch (IOException e) {
                throw new TigerTestEnvException((Throwable)e, "Unable to start helm chart '%s'!", new Object[]{this.tigerServer.getServerId()});
            }
        }, this.executor).thenApplyAsync(process -> {
            process.onExit().thenApply(this.processes::remove);
            this.processes.add((Process)process);
            return process;
        });
    }

    public <T> T getSafely(CompletableFuture<T> future, String cmdText) {
        try {
            return future.get();
        }
        catch (ExecutionException e) {
            throw new TigerTestEnvException((Throwable)e, "Error while executing command %s", new Object[]{cmdText});
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new TigerTestEnvException((Throwable)e, "Interruption received while executing command %s", new Object[]{cmdText});
        }
    }

    public void addLogForPod(String podName, String nameSpace) {
        List<String> statusLines = this.getKubernetesStatusLines(nameSpace);
        Optional<String> optPod = this.getStatusLineForPod(statusLines, podName);
        optPod.ifPresent(s -> this.startLog(s.substring(0, s.indexOf(" "))));
    }

    private void startLog(String podName) {
        this.log.info("Starting log for pod {}", (Object)podName);
        ProcessBuilder processBuilder = new ProcessBuilder(new String[0]).command(this.kubeCtlCommand, "logs", podName, "-n", this.tigerServer.getConfiguration().getHelmChartOptions().getNameSpace(), "-f").redirectErrorStream(true);
        this.spinUpNewExternalProcess(processBuilder).thenAccept(process -> {
            new TigerStreamLogFeeder("Pod " + podName, this.log, process.getInputStream(), Level.INFO);
            new TigerStreamLogFeeder("Pod " + podName, this.log, process.getErrorStream(), Level.ERROR);
        });
    }

    private void applyEnvPropertiesToProcess(ProcessBuilder processBuilder) {
        processBuilder.environment().putAll(this.tigerServer.getEnvironmentProperties().stream().map(str -> str.split("=", 2)).filter(ar -> ((String[])ar).length == 2).collect(Collectors.toMap(ar -> ar[0].trim(), ar -> ar[1].trim())));
    }

    public void stopAllProcesses() {
        for (Process process : this.processes) {
            if (!process.isAlive()) continue;
            if (this.log.isInfoEnabled()) {
                this.log.info("Destroying process calling kubernetes/helm {} ({})", (Object)process.pid(), (Object)process.info().commandLine().orElse(""));
            }
            process.destroy();
        }
    }

    public void setWorkingDirectory(String workingDirectory) {
        this.workingDirectory = workingDirectory;
    }
}

