(ns auth-utils.core
  (:require[clojure.tools.logging :as log]
           [buddy.sign.jws :only [unsign sign] :as jws]
           [clostache.parser :as clostache]
           [clojure.java.io :as io]
           [ring.util.codec :as codec]
           [environ.core :refer [env]]
           [clojure.data.json :as json]
           [buddy.core.codecs :as buddy-codec])
  (:import [com.onelogin AccountSettings]
           [com.onelogin.saml Utils]
           [com.pingidentity.opentoken Agent]
           [com.pingidentity.opentoken AgentConfiguration]
           [com.onelogin.saml Response]
           [org.apache.xml.security Init]
           [org.apache.xml.security.utils Constants]
           [org.apache.xml.security.transforms Transforms]
           [org.apache.xml.security.c14n Canonicalizer]
           [org.apache.xml.security.signature XMLSignature]
           [java.security KeyStore]
           [java.io StringWriter]
           [javax.xml.transform.dom DOMSource]
           [javax.xml.transform.stream StreamResult]
           [javax.xml.transform TransformerFactory]))

;; JWT Implementation
(def jwt-secret (or (env :jwt-secret) "digital-connect-framework"))

(defn get-jwt-attrs
  "Gets all attributes from incoming JWT. See [[build-jwt]]"
  [jwt]
  (try
    (json/read-str (buddy-codec/bytes->str (jws/unsign jwt jwt-secret)) :key-fn keyword)
    (catch Exception e
      (log/error "JWT Unsigning failed." e)
      {:error "No attributes - invalid JWT."})))

(defn build-user-jwt
  [subject user-map]
  (jws/sign (json/write-str {:sub subject :user user-map}) jwt-secret))

(defn build-app-jwt
  [subject app-map]
  (jws/sign (json/write-str {:sub subject :app app-map}) jwt-secret))

(defn build-user-and-app-jwt
  [subject user-map app-map]
  (jws/sign (json/write-str {:sub subject :user user-map :app app-map}) jwt-secret))

(defn- jwt-has-sub?
  [attrs]
  (not (nil? (:sub attrs))))

(defn- jwt-has-user?
  [attrs]
  (let [has-user (not (nil? (:user attrs)))
        has-user-id (not (nil? (:id (:user attrs))))]
    (and has-user has-user-id)))

(defn- jwt-has-app?
  [attrs]
  (let [has-app (not (nil? (:app attrs)))
        has-app-id (not (nil? (:id (:app attrs))))]
    (and has-app has-app-id)))

(defn valid-jwt?
  [jwt]
  (let [attrs (get-jwt-attrs jwt)]
    (and (jwt-has-sub? attrs) (or (jwt-has-user? attrs) (jwt-has-app? attrs)))))

;; All of the following is to support SAML/SSO for Club and Insurance Portal.
(def ^:private saml-template (slurp (io/resource "saml.xml")))

;; prod-cert
(def ^:private prod-cert "MIIFDTCCBHagAwIBAgIKK5ujuQAAAAGoZDANBgkqhkiG9w0BAQUFADBxMQswCQYDVQQGEwJVUzEwMC4GA1UEChMnQUFBIE5vcnRoZXJuIENhbGlmb3JuaWEgTmV2YWRhIEFuZCBVdGFoMTAwLgYDVQQDEydBQUEgTm9ydGhlcm4gQ2FsaWZvcm5pYSBOZXZhZGEgQW5kIFV0YWgwHhcNMTUwMjA2MTcwNzUxWhcNMTcwMjA1MTcwNzUxWjCBkzELMAkGA1UEBhMCVVMxEDAOBgNVBAgTB0FyaXpvbmExETAPBgNVBAcTCEdsZW5kYWxlMTAwLgYDVQQKEydBQUEgTm9ydGhlcm4gQ2FsaWZvcm5pYSBOZXZhZGEgQW5kIFV0YWgxCzAJBgNVBAsTAklUMSAwHgYDVQQDExdiMmMtc3NvLmVudC5ydC5jc2FhLmNvbTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBANivGqGMB4OuMtJgrrSp8d9X9BRaXhJijxIsRrlpwzi3vfQ+JaJiiedmR86H+AOP2tkP19p4Wfp7MorSWDNZtxX5loOqnNJO4LI0wgk1wYyZv21d+LdnMyZCDRe91di+9DeYzkOhO2zS/yX6pXdnqu4Pq1Ch9Q6H/VQ9ZbCIjh6y0PUCcPk3QbqNj+ww8S+qtR/z+R8bltegPCHjEkLyJsQtWnCuTX2Gp/CBSNKbDoSgPID8QwC4/1rklnS6n/6Bf4uynmsWLMhWe4CBP9GzwjhM8FlO2tqwZhOIuUsTYk1LE3sUCsdyopVd5zTvHUZtmvQ1/h9/9F9yCoajXUGjxZ0CAwEAAaOCAgMwggH/MAsGA1UdDwQEAwIFoDATBgNVHSUEDDAKBggrBgEFBQcDATB4BgkqhkiG9w0BCQ8EazBpMA4GCCqGSIb3DQMCAgIAgDAOBggqhkiG9w0DBAICAIAwCwYJYIZIAWUDBAEqMAsGCWCGSAFlAwQBLTALBglghkgBZQMEAQIwCwYJYIZIAWUDBAEFMAcGBSsOAwIHMAoGCCqGSIb3DQMHMB0GA1UdDgQWBBT8or0/cJA6I3fAbwNIC8Scx7LjRTAfBgNVHSMEGDAWgBTYXmJ3g0x9NVXVs1a2up+7dqQ9ajBqBgNVHR8EYzBhMF+gXaBbhllodHRwOi8vcmV2b2NhdGlvbi5jc2FhLmNvbS9jZXJ0ZGF0YS9BQUElMjBOb3J0aGVybiUyMENhbGlmb3JuaWElMjBOZXZhZGElMjBBbmQlMjBVdGFoLmNybDCBkQYIKwYBBQUHAQEEgYQwgYEwfwYIKwYBBQUHMAKGc2h0dHA6Ly9yZXZvY2F0aW9uLmNzYWEuY29tL2NlcnRkYXRhL1AwMUNBVzAwMy5lbnQucnQuY3NhYS5jb21fQUFBJTIwTm9ydGhlcm4lMjBDYWxpZm9ybmlhJTIwTmV2YWRhJTIwQW5kJTIwVXRhaC5jcnQwIQYJKwYBBAGCNxQCBBQeEgBXAGUAYgBTAGUAcgB2AGUAcjANBgkqhkiG9w0BAQUFAAOBgQC+2v+WbR6tgK9HWVcchgwsZqz5XVTx4yX5vn/TibrmEGUTINJE3fhGZ/66X3poClqCEaYnhL04ICzG5L92vMPleZ5G1OY267de4vMIQWClwrwrl6T0bm8vj8KeuVAFGiSzQZuB5e5c1LTHYpZQdXqtV2M/50BUt4aUE0QrFwQdTg==")

;; test-cert
(def ^:private test-cert "MIIChjCCAe+gAwIBAgIBADANBgkqhkiG9w0BAQ0FADBgMQswCQYDVQQGEwJ1czEQMA4GA1UECAwHQXJpem9uYTEeMBwGA1UECgwVQ1NBQSBEaWdpdGFsIFNlcnZpY2VzMR8wHQYDVQQDDBZjc2FhLWluc3VyYW5jZS5hYWEuY29tMB4XDTE2MTAxMTE2NDIzMVoXDTE3MTAxMTE2NDIzMVowYDELMAkGA1UEBhMCdXMxEDAOBgNVBAgMB0FyaXpvbmExHjAcBgNVBAoMFUNTQUEgRGlnaXRhbCBTZXJ2aWNlczEfMB0GA1UEAwwWY3NhYS1pbnN1cmFuY2UuYWFhLmNvbTCBnzANBgkqhkiG9w0BAQEFAAOBjQAwgYkCgYEA2EQZTMkSYs3tJRvsAgGXcf27TjwfxiDVp2GK0N7Lwbauy+7l/HbZ+zHWHsAbOvDnyrX6uPLlfXXUCyy5gqePNILUESpVh7yQ/JFXVmH7132LJQyJK3PUDtNWRBqjSpSSgyCynjzbyvzgLnlSqXC+f7167nDevBmgkLFQwyoXgq8CAwEAAaNQME4wHQYDVR0OBBYEFLJUJOnjRvrdcZi0q2Nv1i/+oBezMB8GA1UdIwQYMBaAFLJUJOnjRvrdcZi0q2Nv1i/+oBezMAwGA1UdEwQFMAMBAf8wDQYJKoZIhvcNAQENBQADgYEAl3o1mfCllC6dU2FagqfqptqVcuiULeLSEfNcybGuwDgiXAGOrIcObYOIMH8z37uqCuzTIwN9+t/d+MMATnf0OuYxo08T+fi1zSDkfJ77xeybU6F4uP/H3H3ilRn2OpJ5bbH4W5T8vA4JLlYnrflIlF/DpNYPWjFt7Meeq0HSikc=")

(def ^:private test-cert-private-key "MIICdwIBADANBgkqhkiG9w0BAQEFAASCAmEwggJdAgEAAoGBAL0f4Vj8/O6Y4hGIF+nPvdz/Bmz7AeonoTGRPR2P4z+0zdvO7h2szjAnhxH/ujGcdJE88VSa8pCwfDHVAaH5CGfIpCQwwzJ9i/iaI98pag8ty8DzTm6D9YAtttKwvYbAjqZxs/W9SlNOU5Ygm3Rf+6Pi0WdOWgKUG7AeZfZA8yS/AgMBAAECgYEArL04o4H3N6qWGbM3PEyNuxOH9/RPrGJj/ZUNwDmTM7XdrN3VaW9TqHICEbOEihm/8oQ6XK9lzDgWR6Gpaxq9zBHhzOgzXzx9zdonCmpAR75iatJ5ff/0E8a6nU/5nBce8wMCq8erEcS7AtXm905vhbKsDHkeGaH1ygtGzeOgDekCQQDwDiMTlMUhSsBO63GrqxQ3hOlRIVzN49T38LSJgMP/X18sOyQdMafIPIC45wBpzH/5X/3xBD9juOEqurxEjAubAkEAya+5tb0829Q9N22grHdXT4OWghP0S0ZYpQOPqur0l6UPAc6dbk2eQQWqmUb0jnl4E6wicoYQlIlMgM0Y7wM3rQJBAOHZuyCiGHHBHXtiwphrpeKyNr1w8Rp4nxcCrSitMDnkpes8XTbmT9/xJH5SP65Ki6C4snd+spEFETudYqHvTL8CQHurjxqLsVeyqt7kEBPxEc6mB5I3niLSPeThbCUH8XdTNQKMqShL2/BPAy79vVPn0+NsN6s13dDW59xK5zEYLfkCQAhxGybpe2riTk+tfw2rTfwKW2EvnoUpJcF12852ygXWROAtgOpArJ4rJ/apY4IeQ8PZLAIlRQ0UmA4T1Rta6GQ=")

(defn- is-test-cert?
  []
  (or (env :use-test-cert) "true"))

(defn- get-cert
  []
  (if (= (is-test-cert?) "true")
    (do (log/debug "Test CERT is in play!") test-cert)
    (do (log/debug "Prod CERT is in play!") prod-cert)))

(defn- saml-xml-document
  [xml]
  (Utils/loadXML xml))

(defn- get-opentoken-attrs
  [opentoken]
  (log/debug "Reading attrs from open token: " opentoken)
  (let [config (AgentConfiguration. true)
        agent (Agent. config)
        attrs (.readToken agent opentoken)]
    (clojure.walk/keywordize-keys (into {} attrs))))

(defn- build-opentoken
  [values]
  (log/debug "Writing values to open token: " values)
  (let [config (AgentConfiguration. true)
        agent (Agent. config)
        attrs (java.util.HashMap. (clojure.walk/stringify-keys values))]
    (.writeToken agent attrs)))

(defn- get-saml-response
  [saml url]
  (let [acct-instance (doto (new AccountSettings) (.setCertificate (get-cert)))]
    (try
      (new Response acct-instance saml url)
      (catch Exception e
        (log/error "SAML loading failed." e)
        nil))))

(defn- valid-saml?
  [response]
  (if (.isValid response nil)
    true
    (do (log/error "SAML Validation failed." (.getError response)) false)))

(defn- get-saml-attrs
  "Gets all attributes for incoming SAML. See [[build-saml]]"
  [saml url]
  (let [response (get-saml-response saml url)]
    (if (valid-saml? response)
      (.getAttributes response)
      {:error "No attributes - invalid SAML."})))

(defn- create-saml-xml
  [user-id location]
  (clostache/render-resource "saml.xml" {:customerRegistrationID user-id :destination location}))

(defn- load-key-store [keystore-filename keystore-password]
  (with-open [is (clojure.java.io/input-stream keystore-filename)]
    (doto (KeyStore/getInstance "JKS")
      (.load is (.toCharArray keystore-password)))))

(defn- xml-doc-to-string
  [doc]
  (let [source (new DOMSource doc)
        writer (new StringWriter)
        result (new StreamResult writer)
        factory (TransformerFactory/newInstance)
        transformer (.newTransformer factory)]
    (.transform transformer source result)
    (.toString writer)))

(defn- sign-saml
  [message]
  (Init/init)
  (let [ks (load-key-store (io/resource "framework-keystore.jks") "csaa123")
        private-key (.getKey ks "1" (.toCharArray "csaa123"))
        cert (.getCertificate ks "1")
        sig-algo XMLSignature/ALGO_ID_SIGNATURE_RSA]
    (let [xmldoc (saml-xml-document message)
          transforms (doto (new Transforms xmldoc)
                       (.addTransform Transforms/TRANSFORM_ENVELOPED_SIGNATURE)
                       (.addTransform Transforms/TRANSFORM_C14N_EXCL_OMIT_COMMENTS))
          sig (new XMLSignature xmldoc nil sig-algo Canonicalizer/ALGO_ID_C14N_EXCL_OMIT_COMMENTS)]
      (let [root (.getDocumentElement xmldoc)
            assertions (.getLastChild root)
            status (.getPreviousSibling assertions)]
         (.insertBefore root (.getElement sig) status))
      (doto sig
        (.addDocument "#qPYNVEhw3Ugz6iIsZvJ3m8GS-_K" transforms Constants/ALGO_ID_DIGEST_SHA1)
        (.addKeyInfo cert)
        (.addKeyInfo (.getPublicKey cert))
        (.sign private-key))
      (xml-doc-to-string xmldoc))))

(defn- encode-saml
  [saml-to-encode]
  (let [base64d (codec/base64-encode (.getBytes saml-to-encode))]
    base64d))

(defn- build-saml
  "Builds a saml for a user for a specific location. See [[sign-saml]]"
  [user-id location]
  (encode-saml (sign-saml (create-saml-xml user-id location))))
