package consulting.omnia.util.cast;

import java.math.BigDecimal;
import java.text.ParseException;
import java.util.Date;

/**
 * Cast Util.
 * Casts and converts types.
 * 
 * @author Ronaldo Blanc ronaldoblanc at omnia.consulting
 */
public class CastUtil extends AbstractCastUtil{

	// TODO try to convert others types
	
	/**
	 * Casts or converts the value as type.
	 * This method is a "number based" converter.
	 * If you pass a numeric value (number or string) it tries to convert data
	 * from this numeric value. So, if you try to convert a string with a date
	 * like 20150801, you will get undesirable result. 
	 * 
	 * @param value object to cast
	 * @param type class to cast object as
	 * @param <T> returning type
	 * @return if possible returns <code>value</code> cast as class <code>type</code>
	 * @throws ParseException if a value can not be parsed
	 */
	public static <T> T getAs(final Object value, final Class<T> type) throws ParseException {
		if(value == null || type == null) {
			throw CastExceptionUtil.castException("Both value and type must be not null");
		}
		
		if (type.isAssignableFrom(value.getClass())) {
			return castAs(value, type);
		}
		
		if (value instanceof Number) {
			final BigDecimal number = NumberCastUtil.numberToBigDecimal(castAs(value, Number.class));
			return numberToType(number, type);
		} else if (value instanceof Date) {
			final BigDecimal number = NumberCastUtil.numberToBigDecimal(castAs(value, Date.class).getTime());
			return numberToType(number, type);
		} else if (value instanceof Boolean) {
			final Boolean booleanValue = castAs(value, Boolean.class);
			return booleanValue ? numberToType(BigDecimal.ONE, type) : numberToType(BigDecimal.ZERO, type); 
		} else if (value instanceof String) {
			try {
				final BigDecimal number = new BigDecimal(castAs(value, String.class));
				return numberToType(number, type);
			} catch(NumberFormatException nfe) {
				if(Boolean.class.isAssignableFrom(type)) {
					return type.cast(BooleanCastUtil.stringToBoolean(castAs(value, String.class)));
				} else if (Date.class.isAssignableFrom(type)) {
					return type.cast(DateCastUtil.stringToDate(castAs(value, String.class)));
				} else {
					throw CastExceptionUtil.castException(value, type);
				}
			}
		} else if (value instanceof Character) {
			final BigDecimal number = new BigDecimal(castAs(value, Character.class).toString());
			return numberToType(number, type);
		} else {
			throw CastExceptionUtil.castException(value, type);
		}

	}

	public static String numberAsString(final Number number) {
		return NumberCastUtil.numberToString(number);
	}
	
	private static <R> R numberToType(final BigDecimal number, final Class<R> type) {
		if (Number.class.isAssignableFrom(type)) {
			return NumberCastUtil.bigDecimalToNumber(number, type);
		} else if (Boolean.class.isAssignableFrom(type)) {
			return type.cast(BigDecimal.ONE.compareTo(number) == 0);
		} else if (Date.class.isAssignableFrom(type)) {
			return type.cast(new Date(NumberCastUtil.bigDecimalToNumber(number, Long.class)));
		} else if (String.class.isAssignableFrom(type)) {
			return type.cast(number.toPlainString());
		} else {
			throw CastExceptionUtil.castException(number, type);
		}
	}
	
}
