/*
 *
 *  * Copyright (c) 2021. Tap Payments
 *  * @author <a href="mailto:c.dommara@tap.company">Charan Dommara</a>
 *  * Created On: 21 1 2021
 *
 */

package company.tap.commondependencies.ISO8583.models;

import com.google.common.base.Strings;
import company.tap.commondependencies.ISO8583.enums.fields;
import company.tap.commondependencies.ISO8583.exceptions.ISOException;
import company.tap.commondependencies.ISO8583.utils.FixedBitSet;
import company.tap.commondependencies.ISO8583.utils.StringUtil;
import lombok.Data;
import org.apache.commons.codec.binary.Hex;

import java.util.Arrays;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

@Data
public class ISOMessage {
    private TreeMap<Integer, byte[]> dataElements = new TreeMap<>();

    private boolean isNil = true;
    private String message;
    private String mti;
    private byte[] msg;
    private byte[] header;
    private byte[] body;
    private byte[] primaryBitmap;
    private byte[] secondaryBitmap;
    private int msgClass;
    private int msgFunction;
    private int msgOrigin;
    private int len = 0;

    public static ISOMessage NullObject() {
        return new ISOMessage();
    }

    public boolean isNil() {
        return isNil;
    }

    public byte[] getHeader() {
        return header;
    }

    public byte[] getBody() {
        return body;
    }

    /**
     * Get primary bitmap
     *
     * @return returns primary byte array
     * @since 1.0.4-SNAPSHOT
     */
    public byte[] getPrimaryBitmap() {
        return primaryBitmap;
    }

    /**
     * Message length
     *
     * @return returns message length
     */
    public int length() {
        return len;
    }

    /**
     * Get field value in byte array format
     *
     * @param fieldNo field number
     * @return returns field value in byte array format
     */
    public byte[] getField(int fieldNo) {
        if (!dataElements.containsKey(fieldNo))
            return new byte[0];
        return dataElements.get(fieldNo);
    }

    /**
     * Get field value in byte array format
     *
     * @param field field in {@link fields} format
     * @return returns field value in byte array format
     */
    public byte[] getField(fields field) {
        return dataElements.get(field.getNo());
    }

    /**
     * Get field value in string format
     *
     * @param fieldNo  field number
     * @return returns field value in String format
     */
    public String getStringField(int fieldNo)  {
        return getStringField(fields.valueOf(fieldNo));

    }

    /**
     * Get field value in string format
     *
     * @param field    field in {@link fields} format
     * @return returns field value in String format
     */
    public String getStringField(fields field) {

        String temp = new String(getField(field.getNo()));
        return temp;
    }

    /**
     * Set and parse ISO8583 message from buffer
     *
     * @param message         ISO8583 in byte array format
     * @param headerAvailable set true if header is available in buffer
     * @return returns ISO8583 message in ISOMessage type
     * @throws ISOException throws exception
     */
    public ISOMessage setMessage(byte[] message, boolean headerAvailable) throws ISOException {

        isNil = false;

        msg = message;
        len = msg.length;

        int headerOffset = 0;

        if (headerAvailable) {
            headerOffset = 12;
        }
        System.out.println(new String(msg));
        try {

            this.header = Arrays.copyOfRange(msg, 0, headerOffset);
            System.out.println(this.header.length);
            this.body = Arrays.copyOfRange(msg, headerOffset, msg.length);
            System.out.println(this.body.length);
            System.out.println(new String(this.body));
            this.primaryBitmap = Arrays.copyOfRange(body, 4, 20);
            System.out.println(this.primaryBitmap.length);
            System.out.println(new String(this.primaryBitmap));
            this.secondaryBitmap = Arrays.copyOfRange(body, 20, 36);
            System.out.println(this.secondaryBitmap.length);
            System.out.println(new String(this.secondaryBitmap));

            parseHeader();
            parseBody();

        } catch (Exception e) {
            e.printStackTrace();
            throw new ISOException(e.getMessage(), e.getCause());
        }

        return this;
    }

    /**
     * Set and parse ISO8583 message from buffer
     *
     * @param message ISO8583 in byte array format
     * @return returns ISO8583 message in ISOMessage type
     * @throws ISOException throws exception
     */
    public ISOMessage setMessage(byte[] message) throws ISOException {
        return this.setMessage(message, true);
    }

    private void parseHeader() {
        if (body.length > 2) {
            mti = new String(Arrays.copyOfRange(body, 0, 4));
            msgClass = Integer.parseInt(mti.substring(1, 2));
            msgFunction = Integer.parseInt(mti.substring(2, 3));
            msgOrigin = Integer.parseInt(mti.substring(3, 4));
        }
    }

    private void parseBody() {
        FixedBitSet pb = new FixedBitSet(128);
        pb.fromHexString(new String(primaryBitmap) + new String(this.secondaryBitmap));
        int offset = 20;

        for (int o : pb.getIndexes()) {

            fields field = fields.valueOf(o);
            if (field.isFixed()) {
                int len = field.getLength();
                addElement(field, Arrays.copyOfRange(body, offset, offset + len));
                offset += len;
            } else {

                int formatLength = 1;
                switch (field.getFormat()) {
                    case "LL":
                        formatLength = 2;
                        break;
                    case "LLL":
                        formatLength = 3;
                        break;
                }

                int flen = Integer.valueOf(new String(Arrays.copyOfRange(body, offset, offset + formatLength)));
                offset = offset + formatLength;

                addElement(field, Arrays.copyOfRange(body, offset, offset + flen));

                offset += flen;
            }

        }
    }

    private void addElement(fields field, byte[] data) {
        System.out.println(field.getNo() + "-" +  new String(data));
        dataElements.put(field.getNo(), data);
    }


    /**
     * Get EntrySet
     *
     * @return returns data elements entry set
     */
    public Set<Map.Entry<Integer, byte[]>> getEntrySet() {
        return dataElements.entrySet();
    }

    /**
     * Check Field exists by {@link fields} enum
     *
     * @param field field enum
     * @return Returns true if field has value in message
     */
    public boolean fieldExits(fields field) {
        return fieldExits(field.getNo());
    }

    /**
     * Check Field exists field number
     *
     * @param no field number
     * @return Returns true if field has value in message
     */
    public boolean fieldExits(int no) {
        return dataElements.containsKey(no);
    }

    /**
     * Get Message MTI
     *
     * @return returns MTI in String format
     */
    public String getMti() {
        return mti;
    }

    /**
     * Get message class
     *
     * @return returns message class
     */
    public int getMsgClass() {
        return msgClass;
    }

    /**
     * Get message function
     *
     * @return returns message function
     */
    public int getMsgFunction() {
        return msgFunction;
    }

    /**
     * Get message origin
     *
     * @return returns message origin
     */
    public int getMsgOrigin() {
        return msgOrigin;
    }


    /**
     * Convert ISOMessage to String
     *
     * @return ISOMessage in String format
     */
    public String toString() {
        return Strings.padStart(String.valueOf(msg.length), 4, '0') + new String(msg);
    }

    /**
     * Convert all fields in String format
     *
     * @return returns strings of fields
     */
    public String fieldsToString() {
        StringBuilder stringBuilder = new StringBuilder();

        stringBuilder.append("\r\n");
        for (Map.Entry<Integer, byte[]> item :
                dataElements.entrySet()) {
            stringBuilder
                    .append(fields.valueOf(item.getKey()).name())
                    .append(" : ")
                    .append(new String(item.getValue()))
                    .append("\r\n");
        }
        stringBuilder.append("\r\n");
        return stringBuilder.toString();
    }

    /**
     * Clean up message
     */
    public void clear() {

        Arrays.fill(header, (byte) 0);
        Arrays.fill(body, (byte) 0);
        Arrays.fill(primaryBitmap, (byte) 0);

        message = null;
        header = null;
        body = null;
        primaryBitmap = null;

    }

    public String toHexString() {
        return Strings.padStart(String.valueOf(msg.length), 4, '0') + String.valueOf(Hex.encodeHex(msg));
    }
}
