package company.tap.commondependencies.Operator;

import company.tap.commondependencies.Constants.CommonConstants;
import company.tap.commondependencies.Errors.Errors;
import company.tap.commondependencies.Errors.IErrorServices;
import company.tap.commondependencies.Helpers.CommonValidationHelper;
import company.tap.commondependencies.Helpers.DataUtilities;
import company.tap.commondependencies.Helpers.RestAPI;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.http.*;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import static org.springframework.http.MediaType.APPLICATION_JSON;

@Service
class OperatorHelper implements IOperatorHelper {
    /*
        Author - Charan Dommara
        Created On - 02 SEP 202
        The class is used to retrieve operator info and validate operator
    */
    private static final Logger logger = LogManager.getLogger(OperatorHelper.class);
    private final String tapApiUrl = DataUtilities.getInstance().getUrlTapApiV2();
    final
    IErrorServices _errorServices;


    private static int expiryChecker=1000;
    Map<String,Long> operatorExpiry =new HashMap<>();
    Map<String,PrivateOperator> privateOperatorMap=new HashMap<>();

    public OperatorHelper(IErrorServices _errorServices) {
        this._errorServices = _errorServices;
    }

    @Override
    public PrivateOperator GetOperatorInfoByKey(String key) {
        try {
            if(--expiryChecker<=0) removeExpiredOperators();
            PrivateOperator privateOperator = privateOperatorMap.get(key);
            Long expiresAt = operatorExpiry.get(key);
            if(null!=privateOperator && null!=expiresAt && expiresAt>(new Date()).getTime()) {
                return privateOperator;
            }
            HttpHeaders headersCA = new HttpHeaders();
            headersCA.setContentType(APPLICATION_JSON);
            headersCA.set("Authorization", "Bearer " + key);
            HttpEntity<String> entityToken = new HttpEntity<>(headersCA);
            privateOperator = RestAPI.getInstance()
                    .exchange(tapApiUrl + "operator/private/info", HttpMethod.GET, entityToken, PrivateOperator.class)
                    .getBody();
            privateOperatorMap.put(key,privateOperator);
            operatorExpiry.put(key,(new Date()).getTime()+(CommonConstants.CACHE_EXPIRY));
            return privateOperator;

        } catch (Exception e) {
            logger.info("----- GetOperatorInfoByKey Error -----");
            e.printStackTrace();
            logger.error(e);
            return null;
        }
    }

    private void removeExpiredOperators() {
        logger.info("Removing expired operators...");
        expiryChecker=1000;
        long time = (new Date()).getTime();
        for (String key :
                operatorExpiry.keySet()) {
            Long expiryAt = operatorExpiry.get(key);
            if(null!= expiryAt && expiryAt <time) privateOperatorMap.remove(key);
        }
    }

    @Override
    public ResponseEntity<Object> ValidateOperatorByKey(String key, boolean tap, boolean isMerchant) {
        try {
            PrivateOperator operator = GetOperatorInfoByKey(key);
            if (operator == null) {
                return _errorServices.Error(Errors.Api_key_unauthorised.getCode(), Errors.Api_key_unauthorised.toString()
                        , "Api " +
                                "key is unauthorized to perform the operation", HttpStatus.UNAUTHORIZED);
            }
            if (!operator.getStatus().equalsIgnoreCase(DataUtilities.getInstance().getStatusActive())) {
                return _errorServices.Error(Errors.InActive_Id.getCode(), Errors.InActive_Id.toString(), "Api " +
                        "key is unauthorized to perform the operation, Operator InActive", HttpStatus.UNAUTHORIZED);
            }
            if (tap && !CommonValidationHelper.MASTER_MIDS.contains(operator.legacyId)) {
                return _errorServices.Error(Errors.Api_key_unauthorised.getCode(), Errors.Api_key_unauthorised.toString(),
                        "Api " +
                                "key is unauthorized to perform the operation", HttpStatus.UNAUTHORIZED);
            }
            if (isMerchant && !operator.isMerchant) {
                return _errorServices.Error(Errors.Api_key_unauthorised.getCode(), Errors.Api_key_unauthorised.toString(),
                        "Api " +
                                "key is unauthorized to perform the operation", HttpStatus.UNAUTHORIZED);
            }
            return null;
        }
        catch (Exception e){
            logger.info("----- ValidateOperatorByKey Error -----");
            e.printStackTrace();
            logger.error(e);
            return _errorServices.Error(Errors.Internal_server_error.getCode(), Errors.Internal_server_error.toString(),
                    "Failed to validate the key", HttpStatus.INTERNAL_SERVER_ERROR);
        }
    }
}
