/*
 *
 *  * Copyright (c) 2021. Tap Payments
 *  * @author <a href="mailto:c.dommara@tap.company">Charan Dommara</a>
 *  * Created On: 21 1 2021
 *
 */

package company.tap.commondependencies.Helpers;

import company.tap.commondependencies.ApiModels.ApiHeaders;
import company.tap.commondependencies.Constants.CommonConstants;
import company.tap.commondependencies.Models.CreateBy;
import company.tap.commondependencies.Operator.PrivateOperator;
import org.json.JSONObject;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.stereotype.Component;
import org.springframework.web.client.HttpClientErrorException;

import java.net.InetSocketAddress;
import java.net.URI;
import java.sql.Timestamp;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.chrono.HijrahChronology;
import java.time.chrono.HijrahDate;
import java.time.chrono.HijrahEra;
import java.time.chrono.IsoChronology;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoField;
import java.util.*;

import static org.springframework.http.MediaType.APPLICATION_JSON;

@Component
public class CommonHelpersUtils {
    /**
     * Method is used to convert timestamp to date
     *
     * @param timestamp
     * @return
     */
    public static Date convertTimestampToDate(long timestamp) {
        Timestamp timestampDate = new Timestamp(timestamp);
        return timestampDate;
    }

    /**
     * The method is used to Host SocketAddress from base URL
     *
     * @param baseUrl
     * @return
     */
    public static InetSocketAddress getHost(String baseUrl) {
        URI myUri = URI.create(baseUrl);
        int port = myUri.getPort();
        if (port == -1) {
            if (myUri.getScheme().equalsIgnoreCase("http"))
                port = 80;
            else
                port = 443;
        }
        return new InetSocketAddress(myUri.getHost(), port);
    }

    /**
     * The method is used to append minutes to current Date and Time
     *
     * @param minutes
     * @return
     */
    public static Date getExpiryDate(int minutes) {
        long t = new Date().getTime();
        return new Date(t + (minutes * 60000));
    }

    public static Map<String, String> createdBy(CreateBy createBy) {
        Map<String, String> create = new HashMap<>();
        create.put(CommonConstants.CaseSensitiveID.OPERATOR_ID, createBy.operatorId);
        create.put(CommonConstants.CaseSensitiveID.BUSINESS_ID, createBy.businessId);
        create.put(CommonConstants.CaseSensitiveID.SEGMENT_ID, createBy.segmentId);
        create.put(CommonConstants.CaseSensitiveID.PRODUCT_ID, createBy.productId);
        create.put(CommonConstants.CaseSensitiveID.LEGACY_ID, createBy.legacyId);
        return create;
    }

    public static CreateBy createdBy(Map<String, String> created) {
        CreateBy createBy = new CreateBy();
        createBy.operatorId = created.get(CommonConstants.CaseSensitiveID.OPERATOR_ID);
        createBy.businessId = created.get(CommonConstants.CaseSensitiveID.BUSINESS_ID);
        createBy.segmentId = created.get(CommonConstants.CaseSensitiveID.SEGMENT_ID);
        createBy.productId = created.get(CommonConstants.CaseSensitiveID.PRODUCT_ID);
        createBy.legacyId = created.get(CommonConstants.CaseSensitiveID.LEGACY_ID);
        return createBy;
    }


    public static CreateBy getCreatedByHeaders(ApiHeaders headers) throws HttpClientErrorException {
        CreateBy createBy = new CreateBy();
        createBy.operatorId = headers.operatorId;
        createBy.businessId = headers.businessId;
        createBy.segmentId = headers.segmentId;
        createBy.legacyId = headers.merchant_id;
        return createBy;
    }


    public static CreateBy getCreatedByOperator(PrivateOperator operator) throws HttpClientErrorException {
        CreateBy createBy = new CreateBy();
        createBy.operatorId = operator.operatorId;
        createBy.businessId = operator.businessId;
        createBy.segmentId = operator.segmentId;
        createBy.productId = operator.productId;
        createBy.legacyId = operator.legacyId;
        return createBy;
    }

    public static Date convertStringToDate(String date, String pattern) {
        try {
            SimpleDateFormat formatter = new SimpleDateFormat(pattern);
            return formatter.parse(date);
        } catch (ParseException e) {
            return null;
        }
    }

    public static Date convertHijriDateToGregorain(String hijriDate, String pattern) {
        try {
            DateTimeFormatter formatter = DateTimeFormatter.ofPattern(pattern);
            LocalDate dateH = LocalDate.parse(hijriDate, formatter);
            HijrahDate date = HijrahChronology.INSTANCE.date(HijrahEra.AH,
                    dateH.get(ChronoField.YEAR_OF_ERA), dateH.get(ChronoField.MONTH_OF_YEAR),
                    dateH.get(ChronoField.DAY_OF_MONTH));
            return Date.from(IsoChronology.INSTANCE.date(date).atStartOfDay(ZoneId.systemDefault()).toInstant());
        } catch (Exception e) {
            return null;
        }
    }

    public static String convertDateToString(Date date, String pattern) {
        try {
            SimpleDateFormat formatter = new SimpleDateFormat(pattern);
            return formatter.format(date);
        } catch (Exception e) {
            return null;
        }
    }

    public static HttpHeaders getTapHeadersFromKey(String key) {
        JSONObject jsonObject = new JSONObject();
        jsonObject.put("key", key);
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.setContentType(APPLICATION_JSON);
        HttpEntity<String> entityToken = new HttpEntity<>(jsonObject.toString(), httpHeaders);
        String entityResponse = RestAPI.getInstance()
                .exchange("https://api.tap.company/v2/gologin/session", HttpMethod.POST,
                        entityToken,
                        String.class)
                .getBody();
        jsonObject = new JSONObject(entityResponse);
        JSONObject data = jsonObject.getJSONObject("data");
        httpHeaders.set("merchant_id", data.getString("merchant_id"));
        httpHeaders.set("livemode", String.valueOf(data.getBoolean("live_mode")));
        httpHeaders.set("session_token", data.getString("session_token"));
        httpHeaders.set("authorization", "Bearer " + key);
        return httpHeaders;
    }

    public static HttpHeaders convertApiHeadersToHtppHeaders(ApiHeaders apiHeaders) {
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.setContentType(APPLICATION_JSON);
        httpHeaders.set("merchant_id", apiHeaders.merchant_id);
        httpHeaders.set("livemode", String.valueOf(apiHeaders.live_mode));
        httpHeaders.set("session_token", apiHeaders.session_token);
        httpHeaders.set("authorization", "Bearer " + apiHeaders.authorization);
        httpHeaders.set("operatorId", apiHeaders.operatorId);
        httpHeaders.set("businessId", apiHeaders.businessId);
        httpHeaders.set("segmentId", apiHeaders.segmentId);
        httpHeaders.set("appAuthorization", apiHeaders.appAuthorization);
        return httpHeaders;
    }

    public static String convertStringToUpperCamelCase(String str) {
        StringBuilder builder = new StringBuilder(str);
        // Flag to keep track if last visited character is a
        // white space or not
        boolean isLastSpace = true;

        // Iterate String from beginning to end.
        for (int i = 0; i < builder.length(); i++) {
            char ch = builder.charAt(i);

            if (isLastSpace && ch >= 'a' && ch <= 'z') {
                // Character need to be converted to uppercase
                builder.setCharAt(i, (char) (ch + ('A' - 'a')));
                isLastSpace = false;
            } else if (ch != ' ')
                isLastSpace = false;
            else
                isLastSpace = true;
        }

        return builder.toString();
    }

    public static List<String> removeDuplicateList(List<String> list) {
        if (Objects.isNull(list) || list.size() == 0)
            return list;
        Set<String> uniqueBusinesses = new HashSet<>(list);
        list.clear();
        list.addAll(uniqueBusinesses);
        return list;
    }

    public static String getCountryByEntityId(String entityId) {
        entityId = entityId.split("_")[1];
        entityId = entityId.substring(2, 5);
        switch (entityId) {
            case "TwK":
                return "KW";
            case  "ErA":
                return "AE";
            case "UaS":
                return "SA";
            case "YgE":
                return "EG";
            case "RhB":
                return "BH";
            case "NmO":
                return "OM";
            case "TaQ":
                return "QA";
            case  "NbL":
                return "LB";
            case "RoJ":
                return "JO";
            case "AsU":
                return "US";
            case "GkH":
                return "HK";
            case "PgS":
                return "SG";
            case "RbG":
                return "GB";
            case  "NaC":
                return "CA";
            case "NhC":
                return "CN";
            case "DlN":
                return "NL";
            case "LrI":
                return "IE";
            case "LhP":
                return "PH";
            case "PsE":
                return "ES";
            default:
                return null;
        }
    }
}
