/*
 * Copyright (c) 2021. Tap Payments
 * Author: Charan Dommara
 * Created On: 21 9 2021
 */

package company.tap.commondependencies.exceptions;

import company.tap.commondependencies.Errors.Errors;
import company.tap.commondependencies.Errors.IErrorServices;
import company.tap.commondependencies.Helpers.NotificationHelper;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.HttpServerErrorException;

@ControllerAdvice
public class CustomExceptionHandler {
    private final Logger loggerObj = LogManager.getLogger(this.getClass());

    @Autowired
    private IErrorServices errorServices;

    @Value("${app.Name}")
    private String appName;

    @Value("${app.SlackAlertChannel}")
    private String slackAlertChannel;

    // Handling Runtime Exceptions
    @ExceptionHandler({RuntimeException.class})
    public ResponseEntity<Object> handleRuntimeException(Exception ex) {
        ex.printStackTrace();
        loggerObj.error(ex);
        NotificationHelper.sendSlackMessage(appName, ex.getMessage(), slackAlertChannel);
        return errorServices.Error(Errors.Internal_server_error.getCode(), Errors.Internal_server_error.toString(),
                "Internal server error", HttpStatus.BAD_REQUEST);
    }

    // Handling Generic Exceptions
    @ExceptionHandler({Exception.class})
    public ResponseEntity<Object> handleGenericException(Exception ex) {
        ex.printStackTrace();
        loggerObj.error(ex);
        NotificationHelper.sendSlackMessage(appName, ex.getMessage(), slackAlertChannel);
        return errorServices.Error(Errors.Internal_server_error.getCode(), Errors.Internal_server_error.toString(),
                "Something went wrong", HttpStatus.BAD_REQUEST);
    }

    // Handling HttpClientErrorException Exceptions
    @ExceptionHandler({HttpClientErrorException.class})
    public ResponseEntity<Object> handleHttpClientErrorException(HttpClientErrorException e) {
        loggerObj.error(e);
        loggerObj.error(e);
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.setContentType(MediaType.APPLICATION_JSON);
        return new ResponseEntity<>(e.getResponseBodyAsByteArray(), httpHeaders, e.getStatusCode());
    }

    @ExceptionHandler({RequiredFieldsException.class})
    public ResponseEntity<Object> handleRequiredFieldsException(RequiredFieldsException ex) {
        loggerObj.error(ex);
        return errorServices.Error(Errors.Required_fields_missing.getCode(), Errors.Required_fields_missing.toString(),
                ex.getMessage(), HttpStatus.BAD_REQUEST);
    }

    @ExceptionHandler({InvalidDataException.class})
    public ResponseEntity<Object> handleInvalidDataException(InvalidDataException ex) {
        loggerObj.error(ex);
        return errorServices.Error(Errors.Invalid_Data.getCode(), Errors.Invalid_Data.toString(),
                ex.getMessage(), HttpStatus.BAD_REQUEST);
    }

    @ExceptionHandler({InvalidIdException.class})
    public ResponseEntity<Object> handleInvalidIdException(InvalidIdException ex) {
        loggerObj.error(ex);
        return errorServices.Error(Errors.Invalid_Id.getCode(), Errors.Invalid_Id.toString(),
                ex.getMessage(), HttpStatus.BAD_REQUEST);
    }

    @ExceptionHandler({InActiveIdException.class})
    public ResponseEntity<Object> handleInActiveIdException(InActiveIdException ex) {
        loggerObj.error(ex);
        return errorServices.Error(Errors.InActive_Id.getCode(), Errors.InActive_Id.toString(),
                ex.getMessage(), HttpStatus.BAD_REQUEST);
    }

    @ExceptionHandler({ExistsException.class})
    public ResponseEntity<Object> handleExistsException(ExistsException ex) {
        loggerObj.error(ex);
        return errorServices.Error(Errors.Exists.getCode(), Errors.Exists.toString(),
                ex.getMessage(), HttpStatus.BAD_REQUEST);
    }

    @ExceptionHandler({NotFoundException.class})
    public ResponseEntity<Object> handleNotFoundException(NotFoundException ex) {
        loggerObj.error(ex);
        return errorServices.Error(Errors.Not_Found.getCode(), Errors.Not_Found.toString(),
                ex.getMessage(), HttpStatus.NOT_FOUND);
    }

    @ExceptionHandler({InvalidApiKeyException.class})
    public ResponseEntity<Object> handleInvalidApiKeyException(InvalidApiKeyException ex) {
        loggerObj.error(ex);
        return errorServices.Error(Errors.Invalid_API_Key.getCode(), Errors.Invalid_API_Key.toString(),
                ex.getMessage(), HttpStatus.UNAUTHORIZED);
    }

    @ExceptionHandler({InvalidSessionException.class})
    public ResponseEntity<Object> handleInvalidSessionException(InvalidSessionException ex) {
        loggerObj.error(ex);
        return errorServices.Error(Errors.Invalid_session.getCode(), Errors.Invalid_session.toString(),
                ex.getMessage(), HttpStatus.UNAUTHORIZED);
    }

    @ExceptionHandler({InvalidTokenException.class})
    public ResponseEntity<Object> handleInvalidTokenException(InvalidTokenException ex) {
        loggerObj.error(ex);
        return errorServices.Error(Errors.Invalid_token.getCode(), Errors.Invalid_token.toString(),
                ex.getMessage(), HttpStatus.BAD_REQUEST);
    }

    @ExceptionHandler({UnauthorizedException.class})
    public ResponseEntity<Object> handleUnauthorizedException(UnauthorizedException ex) {
        loggerObj.error(ex);
        return errorServices.Error(Errors.Api_key_unauthorised.getCode(), Errors.Api_key_unauthorised.toString(),
                ex.getMessage(), HttpStatus.UNAUTHORIZED);
    }

    @ExceptionHandler({CustomErrorException.class})
    public ResponseEntity<Object> handleCustomErrorException(CustomErrorException ex) {
        ex.printStackTrace();
        loggerObj.error(ex.getMessage());
        return errorServices.Error(ex.getErrors().getCode(), ex.getErrors().toString(),
                ex.getMessage(), ex.getHttpStatus());
    }

    @ExceptionHandler({FailedException.class})
    public ResponseEntity<Object> handleFailedException(FailedException ex) {
        loggerObj.error(ex);
        return errorServices.Error(Errors.Failed.getCode(), Errors.Failed.toString(),
                ex.getMessage(), HttpStatus.SERVICE_UNAVAILABLE);
    }


    @ExceptionHandler({HttpServerErrorException.class})
    public ResponseEntity<Object> handleHttpServerErrorException(HttpServerErrorException e) {
        loggerObj.error(e);
        loggerObj.error(e);
        NotificationHelper.sendSlackMessage(appName, e.getMessage() + " -- " + e.getResponseHeaders().getOrigin(),
                slackAlertChannel);
        HttpHeaders httpHeaders = new HttpHeaders();
        httpHeaders.setContentType(MediaType.APPLICATION_JSON);
        return new ResponseEntity<>(e.getResponseBodyAsByteArray(), httpHeaders, e.getStatusCode());
    }
}
