/*
 *
 *  * Copyright (c) 2021. Tap Payments
 *  * @author <a href="mailto:c.dommara@tap.company">Charan Dommara</a>
 *  * Created On: 21 1 2021
 *
 */

package company.tap.commondependencies.ISO8583.builders;


import com.google.common.base.Strings;
import company.tap.commondependencies.ISO8583.enums.MessageFunction;
import company.tap.commondependencies.ISO8583.enums.MessageOrigin;
import company.tap.commondependencies.ISO8583.enums.ProcessCodes;
import company.tap.commondependencies.ISO8583.enums.fields;
import company.tap.commondependencies.ISO8583.exceptions.ISOException;
import company.tap.commondependencies.ISO8583.interfaces.DataElement;
import company.tap.commondependencies.ISO8583.interfaces.MessagePacker;
import company.tap.commondependencies.ISO8583.interfaces.ProcessCode;
import company.tap.commondependencies.ISO8583.models.ISOMessage;
import company.tap.commondependencies.ISO8583.utils.ByteArray;
import company.tap.commondependencies.ISO8583.utils.FixedBitSet;

import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Map;
import java.util.TreeMap;

public abstract class BaseMessageClassBuilder<T> implements DataElement<T>, MessagePacker<T>, ProcessCode<T> {

    private String version;
    private String messageClass;
    private String messageFunction = "0";
    private String messageOrigin = "0";
    private String processCode;
    private TreeMap<Integer, byte[]> dataElements = new TreeMap<>();
    private String header;
    private byte paddingByte = 0xF;
    private boolean leftPadding = false;

    public BaseMessageClassBuilder(String version, String messageClass) {
        this.version = version;
        this.messageClass = messageClass;
    }

    public ISOMessage build() throws ISOException {
        ISOMessage finalMessage = new ISOMessage();
        finalMessage.setMessage(buildBuffer(), this.header != null);
        return finalMessage;
    }

    private byte[] buildBuffer() {
        FixedBitSet primaryBitmap = new FixedBitSet(64);
        ByteArray dataBuffer = new ByteArray();
        FixedBitSet secondaryBitmap = new FixedBitSet(64);
        boolean sBit = false;
        for (Map.Entry<Integer, byte[]> elem : dataElements.entrySet()) {
            if (elem.getKey() > 64) {
                sBit = true;
                secondaryBitmap.flip(elem.getKey() - 65);
            } else
                primaryBitmap.flip(elem.getKey() - 1);

            dataBuffer.append(elem.getValue());
        }
        if (sBit) {
            primaryBitmap.flip(0);
            dataElements.put(fields.F1_SecondaryBitmap.getNo(), secondaryBitmap.toHexString().toUpperCase().getBytes());
            dataBuffer.prepend(secondaryBitmap.toHexString().toUpperCase().getBytes());
        }

        dataBuffer.prepend(primaryBitmap.toHexString().toUpperCase().getBytes());
        dataBuffer.prepend((version + messageClass + messageFunction + messageOrigin).getBytes());

        if (header != null)
            dataBuffer.prepend(header.getBytes());

        return dataBuffer.array();
    }

    public DataElement<T> setHeader(String header) {
        this.header = header;
        return this;
    }

    @Override
    public DataElement<T> setField(int no, byte[] value) throws ISOException {
        setField(fields.valueOf(no), value);
        return this;
    }

    @Override
    public DataElement<T> setField(fields field, byte[] value) throws ISOException {
        return setField(field, value, value.length);
    }

    public DataElement<T> setField(fields field, byte[] value, int valueLength) throws ISOException {

        byte[] fValue = value;
        if (value == null)
            throw new ISOException(field.name() + " is Null");

        //length check and padding
        if (field.isFixed()) {
            if (field.isValidate()) {
                if (field.getLength() != valueLength)
                    throw new ISOException("Invalid length at field  - " + field.getNo());
            }
            if (field.getLength() > valueLength && (field.getType().equals("an") || field.getType().equals("ans"))) {
                String paddedValue = String.format("%-" + field.getLength() + "s", new String(fValue));
                fValue = paddedValue.getBytes(StandardCharsets.ISO_8859_1);
            }
        } else {
            int dLen = fValue.length;
            switch (field.getType()) {
                case "z":
                    if (dLen > field.getLength())
                        fValue = Arrays.copyOfRange(fValue, fValue.length - field.getLength(), fValue.length);
                    dLen = fValue.length * 2;
                    break;
            }

            ByteArray valueBuffer = new ByteArray();
            valueBuffer.append(fValue);

            switch (field.getFormat()) {
                case "LL":
                    String lengthLL = Strings.padStart(String.valueOf(valueLength), 2, '0');
                    valueBuffer.prepend(lengthLL.getBytes());
                    break;
                case "LLL":
                    String lengthLLL = Strings.padStart(String.valueOf(dLen), 3, '0');
                    valueBuffer.prepend(lengthLLL.getBytes());
                    break;
            }

            fValue = valueBuffer.array();
            valueBuffer.clear();
        }
        dataElements.put(field.getNo(), fValue);
        return this;
    }

    private byte[] padding(fields field, byte[] value, byte[] fValue) {
        byte[] fixed = new byte[(int) Math.ceil(field.getLength() / 2) * 2];

        if (leftPadding) {
            leftPad(value, fValue, fixed);
        } else {
            rightPad(value, fValue, fixed);
        }
        fValue = fixed;
        return fValue;
    }

    private void leftPad(byte[] value, byte[] fValue, byte[] fixed) {
        for (int i = 0; i < fValue.length; i++) {
            fixed[i] = fValue[i];
        }
        fixed[0] = (byte) (fixed[0] + (paddingByte << 4));
    }

    private void rightPad(byte[] value, byte[] fValue, byte[] fixed) {
        for (int i = 0; i < fValue.length; i++) {
            fixed[i] = (byte) ((fValue[i] & 0x0F) << 4);
            if (i + 1 < value.length)
                fixed[i] += (fValue[i + 1] & 0xF0) >> 4;
        }
        fixed[fValue.length - 1] = (byte) (fixed[fValue.length - 1] + paddingByte);
    }

    public DataElement<T> setField(int no, String value) throws ISOException {
        setField(fields.valueOf(no), value);
        return this;
    }

    public DataElement<T> setField(fields field, String value) throws ISOException {
        byte[] bytes = value.getBytes(StandardCharsets.ISO_8859_1);
        setField(field, bytes, bytes.length);
        return this;
    }

    public DataElement<T> mti(MessageFunction mFunction, MessageOrigin mOrigin) {
        this.messageFunction = mFunction.getCode();
        this.messageOrigin = mOrigin.getCode();
        return this;
    }

    @Override
    public MessagePacker<T> setLeftPadding(byte character) {
        this.leftPadding = true;
        this.paddingByte = character;
        return this;
    }

    @Override
    public MessagePacker<T> setRightPadding(byte character) {
        this.leftPadding = false;
        this.paddingByte = character;
        return this;
    }

    public DataElement<T> processCode(String code) throws ISOException {
        this.processCode = code;
        this.setField(fields.F3_ProcessCode, this.processCode);
        return this;
    }

    public DataElement<T> processCode(ProcessCodes.TTC_100 ttc) throws ISOException {
        this.processCode = ttc.getCode() + ProcessCodes.ATC.Default.getCode() + ProcessCodes.ATC.Default.getCode();
        this.setField(fields.F3_ProcessCode, this.processCode);
        return this;
    }

    public DataElement<T> processCode(ProcessCodes.TTC_100 ttc, ProcessCodes.ATC atcFrom, ProcessCodes.ATC atcTo) throws ISOException {
        this.processCode = ttc.getCode() + atcFrom.getCode() + atcTo.getCode();
        this.setField(fields.F3_ProcessCode, this.processCode);
        return this;
    }

    public DataElement<T> processCode(ProcessCodes.TTC_200 ttc) throws ISOException {
        this.processCode = ttc.getCode() + ProcessCodes.ATC.Default.getCode() + ProcessCodes.ATC.Default.getCode();
        this.setField(fields.F3_ProcessCode, this.processCode);
        return this;
    }

    public DataElement<T> processCode(ProcessCodes.TTC_200 ttc, ProcessCodes.ATC atcFrom, ProcessCodes.ATC atcTo) throws ISOException {
        this.processCode = ttc.getCode() + atcFrom.getCode() + atcTo.getCode();
        this.setField(fields.F3_ProcessCode, this.processCode);
        return this;
    }

    public DataElement<T> setFields(Map<Integer, String> DataElements) throws ISOException {
        for (Map.Entry<Integer, String> set : DataElements.entrySet()) {
            setField(fields.valueOf(set.getKey()), set.getValue());
        }
        return this;
    }
}

