package com.turbospaces.web3j;

import java.util.Map.Entry;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.X509TrustManager;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.config.AbstractFactoryBean;
import org.web3j.protocol.Web3j;
import org.web3j.protocol.http.HttpService;

import com.google.common.net.HostAndPort;
import com.turbospaces.boot.Bootstrap;
import com.turbospaces.boot.BootstrapAware;
import com.turbospaces.common.SSL;
import com.turbospaces.ups.PlainServiceInfo;
import com.turbospaces.ups.UPSs;

import okhttp3.OkHttpClient;
import okhttp3.logging.HttpLoggingInterceptor;

public class Web3jFactoryBean extends AbstractFactoryBean<Web3j> implements BootstrapAware {
    private final Logger log = LoggerFactory.getLogger( getClass() );
    private PlainServiceInfo serviceInfo;
    private Bootstrap bootstrap;

    public Web3jFactoryBean() {
        super();
    }
    public Web3jFactoryBean(PlainServiceInfo serviceInfo) throws Exception {
        this.serviceInfo = Objects.requireNonNull( serviceInfo );
    }
    @Override
    public void setBootstrap(Bootstrap bootstrap) {
        this.bootstrap = bootstrap;
        if ( serviceInfo == null ) {
            serviceInfo = UPSs.findRequiredServiceInfoByName( bootstrap, UPSs.WEB3J );
        }
    }
    @Override
    public Class<?> getObjectType() {
        return Web3j.class;
    }
    @Override
    protected Web3j createInstance() throws Exception {
        OkHttpClient.Builder builder = new OkHttpClient.Builder();
        builder.connectTimeout( bootstrap.props().TCP_CONNECTION_TIMEOUT.get(), TimeUnit.SECONDS );
        builder.readTimeout( bootstrap.props().TCP_SOCKET_TIMEOUT.get(), TimeUnit.SECONDS );

        if ( bootstrap.isDevMode() ) {
            if ( "https".equalsIgnoreCase( serviceInfo.getScheme() ) ) {
                SSL ssl = new SSL();
                ssl.addUntrustedCertificates( HostAndPort.fromHost( serviceInfo.getHost() ) );
                Entry<SSLSocketFactory, X509TrustManager> entry = ssl.buildSSLFactory();
                builder.sslSocketFactory( entry.getKey(), entry.getValue() );
            }
        }

        HttpLoggingInterceptor logging = new HttpLoggingInterceptor( log::debug );
        logging.setLevel( HttpLoggingInterceptor.Level.BODY );
        builder.addInterceptor( logging );

        OkHttpClient okhttp = builder.build();

        String uri = serviceInfo.getUri();
        HttpService httpService = new HttpService( uri, okhttp, false );
        return Web3j.build( httpService );
    }
    @Override
    protected void destroyInstance(Web3j instance) throws Exception {
        if ( instance != null ) {
            instance.shutdown();
        }
    }
}
