package com.turbospaces.ebean;

import java.util.EnumSet;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheStats;
import com.google.common.cache.RemovalCause;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;

import io.ebean.cache.ServerCache;
import io.ebean.cache.ServerCacheStatistics;
import io.ebean.cache.ServerCacheType;
import io.ebeaninternal.server.cache.DefaultServerCacheConfig;

public interface LocalCache extends ServerCache, ClearableCache, RemovableCache {
    String cacheKey();
    ServerCacheType type();
    void cleanUp();

    static ServerCacheStatistics stats(String cacheKey, Cache<String, Object> cache) {
        CacheStats stats = cache.stats();

        ServerCacheStatistics scs = new ServerCacheStatistics();
        scs.setCacheName(cacheKey);
        scs.setEvictCount(stats.evictionCount());
        scs.setHitCount(stats.hitCount());
        scs.setMissCount(stats.missCount());
        scs.setSize((int) cache.size());

        return scs;
    }
    static Cache<String, Object> cache(String cacheKey, DefaultServerCacheConfig cfg) {
        Logger logger = LoggerFactory.getLogger(LocalCache.class);
        return CacheBuilder.newBuilder()
                .maximumSize(cfg.getMaxSize())
                .expireAfterAccess(cfg.getMaxIdleSecs(), TimeUnit.SECONDS)
                .expireAfterWrite(cfg.getMaxSecsToLive(), TimeUnit.SECONDS)
                .recordStats()
                .removalListener(new RemovalListener<Object, Object>() {
                    private final EnumSet<RemovalCause> reasons = EnumSet.of(RemovalCause.EXPIRED, RemovalCause.SIZE);

                    @Override
                    public void onRemoval(RemovalNotification<Object, Object> notification) {
                        if (Objects.nonNull(notification.getCause())) {
                            if (reasons.contains(notification.getCause())) {
                                if (logger.isTraceEnabled()) {
                                    String type = notification.getCause().name().toLowerCase().intern();
                                    logger.trace("onRemoval({}): {} {}", cacheKey, type, notification);
                                }
                            }
                        }
                    }
                }).build();
    }
}
