package com.turbospaces.ebean;

import java.util.EnumSet;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheStats;
import com.google.common.cache.RemovalCause;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;

import io.ebean.cache.ServerCacheStatistics;
import io.ebean.cache.TenantAwareKey;
import io.ebean.config.CurrentTenantProvider;
import io.ebeaninternal.server.cache.DefaultServerCacheConfig;

public class LocalEbeanCache implements LocalCache, Supplier<Cache<Object, Object>> {
    private static EnumSet<RemovalCause> REASONS = EnumSet.of(RemovalCause.EXPIRED, RemovalCause.SIZE);

    private final Logger logger = LoggerFactory.getLogger(getClass());
    private final String cacheKey;
    private final Cache<Object, Object> cache;
    private final TenantAwareKey tenantAwareKey;

    public LocalEbeanCache(String cacheKey, int maxSize, CurrentTenantProvider tenantProvider) {
        this.cacheKey = Objects.requireNonNull(cacheKey);
        this.tenantAwareKey = new TenantAwareKey(tenantProvider);
        this.cache = CacheBuilder.newBuilder().maximumSize(maxSize).recordStats().build();
    }
    public LocalEbeanCache(String cacheKey, DefaultServerCacheConfig cfg, CurrentTenantProvider tenantProvider) {
        this.cacheKey = Objects.requireNonNull(cacheKey);
        this.tenantAwareKey = new TenantAwareKey(tenantProvider);
        this.cache = CacheBuilder.newBuilder()
                .maximumSize(cfg.getMaxSize())
                .expireAfterAccess(cfg.getMaxIdleSecs(), TimeUnit.SECONDS)
                .expireAfterWrite(cfg.getMaxSecsToLive(), TimeUnit.SECONDS)
                .recordStats()
                .removalListener(new RemovalListener<Object, Object>() {
                    @Override
                    public void onRemoval(RemovalNotification<Object, Object> notification) {
                        if (Objects.nonNull(notification.getCause())) {
                            if (REASONS.contains(notification.getCause())) {
                                String type = notification.getCause().name().toLowerCase().intern();
                                logger.trace("onRemoval({}): {} {}", cacheKey, type, notification);
                            }
                        }
                    }
                })
                .build();
    }
    @Override
    public Object get(Object id) {
        Object toReturn = cache.getIfPresent(key(id));
        logger.trace("get by key: {}, value: {}", id, toReturn);
        return toReturn;
    }
    @Override
    public void put(Object id, Object value) {
        logger.trace("put key: {}, value: {}", id, value);
        cache.put(key(id), value);
    }
    @Override
    public void remove(Object id) {
        logger.trace("remove key: {}", id);
        cache.invalidate(key(id));
    }
    @Override
    public int size() {
        return (int) cache.size();
    }
    @Override
    public int hitRatio() {
        return (int) (cache.stats().hitRate() * 100);
    }
    @Override
    public ServerCacheStatistics statistics(boolean reset) {
        CacheStats stats = cache.stats();

        ServerCacheStatistics scs = new ServerCacheStatistics();
        scs.setCacheName(cacheKey);
        scs.setEvictCount(stats.evictionCount());
        scs.setHitCount(stats.hitCount());
        scs.setMissCount(stats.missCount());
        scs.setSize(size());

        return scs;
    }
    @Override
    public void clear() {
        logger.debug("clearing {}(items={})", cacheKey, cache.size());
        cache.invalidateAll();
        cleanUp();
    }
    @Override
    public int onClear() {
        long size = cache.size();
        clear();
        return (int) size;
    }
    @Override
    public void cleanUp() {
        cache.cleanUp();
    }
    @Override
    public Cache<Object, Object> get() {
        return cache;
    }
    @Override
    public String toString() {
        return cacheKey;
    }
    protected Object key(Object key) {
        return tenantAwareKey.key(key).toString();
    }
}
