package com.turbospaces.ebean;

import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.time.StopWatch;
import org.jgroups.blocks.MethodCall;
import org.jgroups.blocks.RequestOptions;
import org.jgroups.blocks.RpcDispatcher;
import org.jgroups.util.RspList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.util.concurrent.FluentFuture;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.MoreExecutors;
import com.turbospaces.boot.Bootstrap;
import com.turbospaces.boot.BootstrapAware;

import io.ebeaninternal.server.cache.DefaultServerCacheConfig;
import io.ebeaninternal.server.cache.DefaultServerQueryCache;
import io.vavr.CheckedFunction0;

public class JgroupsServerQueryCache extends DefaultServerQueryCache implements BootstrapAware {
    private final Logger log = LoggerFactory.getLogger(getClass());
    private final RpcDispatcher dispatcher;
    private Bootstrap bootstrap;

    public JgroupsServerQueryCache(RpcDispatcher dispatcher, DefaultServerCacheConfig config) {
        super(config);
        this.dispatcher = dispatcher;
    }
    @Override
    public void setBootstrap(Bootstrap bootstrap) {
        this.bootstrap = Objects.requireNonNull(bootstrap);
    }
    @Override
    @SuppressWarnings("serial")
    public void clear() {
        onClear();

        //
        // ~ we should not block main thread as such
        //
        StopWatch stopWatch = StopWatch.createStarted();
        FluentFuture.from(bootstrap.globalPlatform().submit(new CheckedFunction0<CompletableFuture<RspList<Object>>>() {
            @Override
            public CompletableFuture<RspList<Object>> apply() throws Throwable {
                CompletableFuture<RspList<Object>> future = CompletableFuture.completedFuture(null);
                if (dispatcher.getChannel().isOpen()) {
                    MethodCall call = new MethodCall(JGroupsCacheManager.METHOD_ON_CACHE_CLEAR, name);
                    future = dispatcher.callRemoteMethodsWithFuture(null, call, RequestOptions.ASYNC());
                    stopWatch.stop();
                }
                return future;
            }
        })).addCallback(new FutureCallback<Object>() {
            @Override
            public void onSuccess(Object result) {
                log.info("cleared query cache {} on remote nodes in: {}", name, stopWatch);
                long time = stopWatch.getTime(TimeUnit.SECONDS);
                if (time > 0) {
                    log.error("clear operation took too long: {}", stopWatch);
                }
            }
            @Override
            public void onFailure(Throwable t) {
                log.error(t.getMessage(), t);
            }
        }, MoreExecutors.directExecutor());
    }
    public void onClear() {
        String idx = bootstrap.props().CLOUD_APP_INSTANCE_INDEX.get();
        int size = size();
        log.info("onClear {}, items: {} on ({})", name, size, idx);
        super.clear();
    }
}
