package com.turbospaces.ebean;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.LocalDate;
import java.time.ZoneOffset;
import java.util.List;

import org.awaitility.Awaitility;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Configurable;
import org.springframework.cloud.service.ServiceInfo;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.annotation.Bean;

import com.turbospaces.boot.AbstractBootstrapAware;
import com.turbospaces.boot.MockCloud;
import com.turbospaces.boot.SimpleBootstrap;
import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.ebean.query.QAccount;
import com.turbospaces.ebean.query.QAccountBalance;
import com.turbospaces.jdbc.HikariDataSourceFactoryBean;
import com.turbospaces.jgroups.JGroupsFactoryBean;
import com.turbospaces.plugins.FlywayBootstrapInitializer;
import com.turbospaces.ups.UPSs;

import io.ebean.CacheMode;
import io.ebean.Database;
import io.ebean.Transaction;

public class CompositeKeyTest {
    private final Integer DELAY_AND_PERIOD_OF_CACHE_EXECUTIONS_MS = 10;

    @Test
    public void works() throws Throwable {
        MockCloud builder = MockCloud.newMock();
        ApplicationConfig cfg = builder.build();
        ApplicationProperties props = new ApplicationProperties(cfg);
        EbeanCacheConfigurer mngr = new DefaultEbeanCacheConfigurer(props);

        mngr.setLocal(UTMTemplate.class);
        mngr.setLocal(Account.class);
        mngr.setLocal(AccountBalance.class);
        mngr.setLocal(AccountBalanceSnapshot.class);
        mngr.setLocal(Account.class, QAccount.alias().balances);
        mngr.setLocal(Account.class, QAccount.alias().utmTemplates);
        mngr.setLocal(AccountBalance.class, QAccountBalance.alias().snapshots);

        //
        // ~ local cache mode
        //
        mngr.setMaxSize(Account.class, 1);
        mngr.setMaxSize(Account.class, QAccount.alias().balances, 1);

        //
        // ~ replicated cache mode
        //
        mngr.setMaxSize(UTMTemplate.class, Byte.SIZE);
        mngr.setMaxSize(AccountBalance.class, 10);
        mngr.setMaxSize(AccountBalance.class, QAccountBalance.alias().snapshots, 10);
        mngr.setMaxSize(AccountBalanceSnapshot.class, 30);
        mngr.setMaxSize(Account.class, QAccount.alias().utmTemplates, Integer.MAX_VALUE);

        //
        // ~ query cache mode
        //
        mngr.setMaxSizeQuery(Account.class, 1);

        SimpleBootstrap bootstrap = new SimpleBootstrap(new ApplicationProperties(cfg), AppConfig.class);
        try {
            bootstrap.withH2(true, bootstrap.spaceName());
            bootstrap.cfg().setDefaultProperty(props.CACHE_METRICS_REPORT_INTERVAL.getKey(), Duration.ofMillis(DELAY_AND_PERIOD_OF_CACHE_EXECUTIONS_MS));

            ServiceInfo ownerUps = UPSs.findRequiredServiceInfoByName(bootstrap, UPSs.H2_OWNER);

            bootstrap.addBootstrapRegistryInitializer(new FlywayBootstrapInitializer(ownerUps, "CORE"));
            ConfigurableApplicationContext applicationContext = bootstrap.run();

            Database ebean = applicationContext.getBean(Database.class);

            Account account = new Account();
            account.setId(System.currentTimeMillis());
            account.setUsername("username_" + account.getId());
            account.setFirstName("f_" + account.getId());
            account.setLastName("l_" + account.getId());
            account.setAge(18);
            account.setBirthDate(LocalDate.now(ZoneOffset.UTC).minusYears(account.getAge()));
            ebean.save(account);

            AccountBalance balance = new AccountBalance(account, "USD");
            balance.setAmount(BigDecimal.ONE);

            AccountBalanceSnapshot snapshot1 = new AccountBalanceSnapshot(balance, LocalDate.now(ZoneOffset.UTC));
            AccountBalanceSnapshot snapshot2 = new AccountBalanceSnapshot(balance, LocalDate.now(ZoneOffset.UTC).plusDays(1));

            balance.getSnapshots().add(snapshot1);
            balance.getSnapshots().add(snapshot2);
            ebean.save(balance);

            QAccountBalance q = new QAccountBalance(ebean);
            q.setId(balance.getPk());
            q.findList();

            try (Transaction tx1 = ebean.beginTransaction()) {
                var query = new QAccountBalance(ebean).usingTransaction(tx1).setUseQueryCache(true);
                query.amount.ge(BigDecimal.ONE);
                query.findList();
            }

            try (Transaction tx2 = ebean.beginTransaction()) {
                var query = new QAccountBalance(ebean).usingTransaction(tx2).setUseQueryCache(true);
                query.amount.ge(BigDecimal.ONE);
                query.findList();
            }

            try (Transaction tx3 = ebean.beginTransaction()) {
                var query = new QAccountBalance(ebean).usingTransaction(tx3).setUseQueryCache(true);
                query.amount.ge(BigDecimal.ONE);
                query.findList();
            }

            Assertions.assertEquals(1, bootstrap.meterRegistry().get("cache.size").tag("cache", "com.turbospaces.ebean.AccountBalance_B").gauge().value());
            Awaitility.await().until(() -> checkQueryMetrics(bootstrap, "query_cache.hits", 2) &&
                    checkQueryMetrics(bootstrap, "query_cache.puts", 1) &&
                    checkQueryMetrics(bootstrap, "query_cache.evicts", 0));

            account.setAge(21);
            ebean.save(account);

            Thread.sleep(10 * 000);
        } finally {
            bootstrap.shutdown();
        }
    }

    @Test
    public void testQueryCacheInvalidation() throws Throwable {
        MockCloud builder = MockCloud.newMock();
        ApplicationConfig cfg = builder.build();

        SimpleBootstrap bootstrap = new SimpleBootstrap(new ApplicationProperties(cfg), AppConfig.class);
        try {
            bootstrap.withH2(true, bootstrap.spaceName());

            ServiceInfo ownerUps = UPSs.findRequiredServiceInfoByName(bootstrap, UPSs.H2_OWNER);
            bootstrap.addBootstrapRegistryInitializer(new FlywayBootstrapInitializer(ownerUps, "CORE"));
            ConfigurableApplicationContext applicationContext = bootstrap.run();

            JpaManager ebean = applicationContext.getBean(JpaManager.class);

            Account account = new Account();
            account.setId(System.currentTimeMillis());
            String username = "username_" + account.getId();
            account.setUsername(username);
            account.setFirstName("f_" + account.getId());
            account.setLastName("l_" + account.getId());
            account.setAge(18);
            account.setBirthDate(LocalDate.now(ZoneOffset.UTC).minusYears(account.getAge()));
            ebean.save(account);

            AccountBalance balance1 = new AccountBalance(account, "USD");
            balance1.setAmount(BigDecimal.ONE);
            ebean.save(balance1);

            AccountBalance balance2 = new AccountBalance(account, "EUR");
            balance2.setAmount(BigDecimal.ONE);
            ebean.save(balance2);

            // update dependant table so account query cache is invalidated
            balance1.setAmount(BigDecimal.valueOf(2));
            balance2.setAmount(BigDecimal.valueOf(3));

            ebean.save(balance1);
            ebean.save(balance2);

            for (int i = 0; i < 10; i++) {
                BigDecimal sum = BigDecimal.ZERO;
                for (Long id : getAccounts(ebean, username)) {
                    for (AccountBalance it : ebean.find(Account.class, id).getBalances()) {
                        sum = sum.add(it.getAmount());
                    }
                }
                Assertions.assertEquals(5, sum.longValue());
            }

            balance1.setAmount(BigDecimal.valueOf(117));
            balance2.setAmount(BigDecimal.valueOf(111));

            try (Transaction tx = ebean.newTransaction()) {
                ebean.save(balance1, tx);
                ebean.save(balance2, tx);
                tx.rollback();
            }

            try (Transaction tx = ebean.newTransaction()) {
                ebean.refresh(balance1);
                ebean.refresh(balance2);
            }

            try (Transaction tx = ebean.newTransaction()) {
                for (int i = 0; i < 3; i++) {
                    QAccountBalance qq = new QAccountBalance(ebean).usingTransaction(tx);
                    qq.setUseQueryCache(false);
                    qq.setBeanCacheMode(CacheMode.OFF);
                    qq.findList();
                }

                tx.setSkipCache(false);
                ebean.refresh(balance1);
                ebean.refresh(balance2);

                balance1.setAmount(BigDecimal.valueOf(7));
                balance1.setCrypto(true);

                balance2.setAmount(BigDecimal.valueOf(11));
                balance2.setCrypto(true);

                ebean.save(balance1, tx);
                ebean.save(balance2, tx);
                ebean.flush();

                balance1.setCrypto(false);
                balance2.setCrypto(false);

                ebean.save(balance1, tx);
                ebean.save(balance2, tx);
                ebean.flush();

                tx.commit();
            }

            // assert account cache is invalidated
            for (int i = 0; i < 10; i++) {
                BigDecimal sum = BigDecimal.ZERO;
                for (Long id : getAccounts(ebean, username)) {
                    for (AccountBalance it : ebean.find(Account.class, id).getBalances()) {
                        sum = sum.add(it.getAmount());
                    }
                }
                Assertions.assertEquals(18, sum.longValue());
            }
        } finally {
            bootstrap.shutdown();
        }
    }

    private static List<Long> getAccounts(JpaManager ebean, String username) throws Exception {
        try (Transaction tx = ebean.newTransaction()) {
            var query = new QAccount(ebean).usingTransaction(tx).setUseQueryCache(true);
            query.username.eq(username);
            return query.findIds();
        }
    }

    private static boolean checkQueryMetrics(SimpleBootstrap bootstrap, String queryName, double expectedCount) {
        try {
            return bootstrap.meterRegistry()
                    .get(queryName)
                    .tag("name", "com.turbospaces.ebean.AccountBalance_Q")
                    .counter()
                    .count() == expectedCount;
        } catch (Exception e) {
            return false;
        }
    }

    @Configurable
    public static class AppConfig extends AbstractBootstrapAware {
        @Bean
        public HikariDataSourceFactoryBean ds() {
            ServiceInfo appUps = UPSs.findRequiredServiceInfoByName(bootstrap, UPSs.H2_APP);
            return new HikariDataSourceFactoryBean(appUps);
        }
        @Bean
        public JGroupsFactoryBean jgroups() {
            return new JGroupsFactoryBean();
        }
        @Bean
        public JGroupCacheManagerFactoryBean cacheManager(JGroupsFactoryBean factory) throws Exception {
            return new JGroupCacheManagerFactoryBean(factory.getObject());
        }
        @Bean
        public EbeanDatabaseConfig ebeanConfig(HikariDataSourceFactoryBean factory) throws Exception {
            EbeanDatabaseConfig cfg = new EbeanDatabaseConfig(factory.getObject(), bootstrap.props());

            cfg.addClass(Account.class);
            cfg.addClass(GameplayInfo.class);
            cfg.addClass(FraudJson.class);
            cfg.addClass(UTMTemplate.class);
            cfg.addClass(AccountBalance.class);
            cfg.addClass(AccountBalanceId.class);
            cfg.addClass(AccountBalanceSnapshot.class);
            cfg.addClass(AccountBalanceSnapshotId.class);

            cfg.setLocal(UTMTemplate.class);
            cfg.setLocal(Account.class);
            cfg.setLocal(AccountBalance.class);
            cfg.setLocal(AccountBalanceSnapshot.class);
            cfg.setLocal(Account.class, QAccount.alias().balances);
            cfg.setLocal(Account.class, QAccount.alias().utmTemplates);
            cfg.setLocal(AccountBalance.class, QAccountBalance.alias().snapshots);

            return cfg;
        }
        @Bean
        public EbeanFactoryBean ebean(EbeanDatabaseConfig config, JGroupCacheManagerFactoryBean cache) throws Exception {
            return new EbeanFactoryBean(config, cache.getObject());
        }
    }
}
