package com.turbospaces.ebean;

import java.lang.reflect.Method;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.apache.commons.lang3.time.StopWatch;
import org.jgroups.blocks.MethodCall;
import org.jgroups.blocks.MethodLookup;
import org.jgroups.blocks.RequestOptions;
import org.jgroups.blocks.RpcDispatcher;
import org.jgroups.util.RspList;

import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.util.concurrent.FluentFuture;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.MoreExecutors;
import com.netflix.archaius.api.Config;
import com.turbospaces.boot.AbstractBootstrapAware;
import com.turbospaces.boot.Bootstrap;

import io.ebean.BackgroundExecutor;
import io.ebean.DatabaseBuilder;
import io.ebean.cache.QueryCacheEntryValidate;
import io.ebean.cache.ServerCache;
import io.ebean.cache.ServerCacheConfig;
import io.ebean.cache.ServerCacheFactory;
import io.ebean.cache.ServerCacheNotification;
import io.ebean.cache.ServerCacheNotify;
import io.ebean.cache.ServerCacheOptions;
import io.ebean.cache.ServerCacheStatistics;
import io.ebean.cache.ServerCacheType;
import io.ebean.config.CurrentTenantProvider;
import io.ebeaninternal.server.cache.DefaultServerCacheConfig;
import io.ebeaninternal.server.cache.DefaultServerQueryCache;
import io.micrometer.core.instrument.Tag;
import io.micrometer.core.instrument.Tags;
import io.micrometer.core.instrument.binder.cache.GuavaCacheMetrics;
import io.vavr.CheckedFunction0;

public class JGroupsCacheManager extends AbstractBootstrapAware implements MethodLookup, CacheManager, ServerCacheNotify {
    public static short METHOD_ON_CACHE_PUT = 1;
    public static short METHOD_ON_CHANGE_REMOVE = 2;
    public static short METHOD_ON_CACHE_CLEAR = 3;
    public static short METHOD_ON_MODIFIED = 4;
    public static short METHOD_ON_CACHE_CLEAR_ALL = 5;

    public static ImmutableMap<Short, Method> METHODS;

    static {
        try {
            ImmutableMap.Builder<Short, Method> b = ImmutableMap.builder();
            b.put(METHOD_ON_MODIFIED, JGroupsCacheManager.class.getMethod("onTablesModify", String.class));
            b.put(METHOD_ON_CACHE_PUT, JGroupsCacheManager.class.getMethod("onCachePut", String.class, byte[].class, byte[].class));
            b.put(METHOD_ON_CHANGE_REMOVE, JGroupsCacheManager.class.getMethod("onCacheRemove", String.class, byte[].class));
            b.put(METHOD_ON_CACHE_CLEAR, JGroupsCacheManager.class.getMethod("onCacheClear", String.class));
            b.put(METHOD_ON_CACHE_CLEAR_ALL, JGroupsCacheManager.class.getMethod("onCacheClearAll", boolean.class));

            METHODS = b.build();
        } catch (NoSuchMethodException err) {
            throw new Error(err);
        }
    }

    private final RpcDispatcher dispatcher;
    private final ConcurrentMap<String, ServerCache> caches;
    private BackgroundExecutor executor;
    private ServerCacheNotify notify;

    public JGroupsCacheManager(RpcDispatcher dispatcher) {
        this.dispatcher = Objects.requireNonNull(dispatcher);
        this.caches = Maps.newConcurrentMap();
    }

    @Override
    public void setBootstrap(Bootstrap bootstrap) throws Throwable {
        super.setBootstrap(bootstrap);
        long intervalMillis = bootstrap.props().CACHE_METRICS_REPORT_INTERVAL.get().toMillis();
        bootstrap.globalPlatform().scheduleWithFixedDelay(() -> caches.forEach((key, value) -> {
            if (key.endsWith(ServerCacheType.QUERY.code())) {
                ServerCacheStatistics stats = value.statistics(true);
                bootstrap.meterRegistry().counter("query_cache.hits", "name", key).increment(stats.getHitCount());
                bootstrap.meterRegistry().counter("query_cache.puts", "name", key).increment(stats.getPutCount());
                bootstrap.meterRegistry().counter("query_cache.evicts", "name", key).increment(stats.getEvictCount());
                bootstrap.meterRegistry().gauge("query_cache.size", Tags.of("name", key), stats.getSize());
                bootstrap.meterRegistry().gauge("query_cache.maxSize", Tags.of("name", key), stats.getMaxSize());
                bootstrap.meterRegistry().gauge("query_cache.hit_ratio", Tags.of("name", key), stats.getHitRatio());
                bootstrap.meterRegistry().counter("query_cache.miss_count", "name", key).increment(stats.getMissCount());
                bootstrap.meterRegistry().counter("query_cache.ttl_count", "name", key).increment(stats.getTtlCount());
                bootstrap.meterRegistry().counter("query_cache.remove_count", "name", key).increment(stats.getRemoveCount());
                bootstrap.meterRegistry().counter("query_cache.clear_count", "name", key).increment(stats.getClearCount());
                bootstrap.meterRegistry().counter("query_cache.idle_count", "name", key).increment(stats.getIdleCount());
            }
        }), intervalMillis, intervalMillis, TimeUnit.MILLISECONDS);

        if (Objects.isNull(executor)) {
            setBackgroundExecutor(new BackgroundExecutor() {
                @Override
                public Future<?> submit(Runnable task) {
                    return bootstrap.globalPlatform().submit(task);
                }
                @Override
                public <T> Future<T> submit(Callable<T> task) {
                    return bootstrap.globalPlatform().submit(task);
                }
                @Override
                public ScheduledFuture<?> scheduleWithFixedDelay(Runnable task, long initialDelay, long delay, TimeUnit unit) {
                    return bootstrap.globalPlatform().scheduleWithFixedDelay(task, initialDelay, delay, unit);
                }
                @Override
                public ScheduledFuture<?> scheduleAtFixedRate(Runnable task, long initialDelay, long period, TimeUnit unit) {
                    return bootstrap.globalPlatform().scheduleAtFixedRate(task, initialDelay, period, unit);
                }
                @Override
                public <V> ScheduledFuture<V> schedule(Callable<V> task, long delay, TimeUnit unit) {
                    return bootstrap.globalPlatform().schedule(task, delay, unit);
                }
                @Override
                public ScheduledFuture<?> schedule(Runnable task, long delay, TimeUnit unit) {
                    return bootstrap.globalPlatform().schedule(task, delay, unit);
                }
                @Override
                public void execute(Runnable task) {
                    bootstrap.globalPlatform().execute(task);
                }
            });
        }
    }
    @Override
    public Method findMethod(short id) {
        return METHODS.get(id);
    }
    @Override
    public void setBackgroundExecutor(BackgroundExecutor executor) {
        this.executor = Objects.requireNonNull(executor);
    }
    @Override
    public ServerCacheFactory create(DatabaseBuilder databaseBuilder, BackgroundExecutor backgroundExecutor) {
        setBackgroundExecutor(backgroundExecutor);
        return this;
    }
    @Override
    public ServerCacheNotify createCacheNotify(ServerCacheNotify cacheNotify) {
        this.notify = Objects.requireNonNull(cacheNotify);
        return this;
    }
    @Override
    @SuppressWarnings("serial")
    public void notify(ServerCacheNotification notification) {
        if (notification.getDependentTables() != null && !notification.getDependentTables().isEmpty()) {
            Set<String> dependentTables = notification.getDependentTables();
            String line = Joiner.on(',').join(dependentTables);
            logger.debug("Publish TableMods - {}", line);

            StopWatch stopWatch = StopWatch.createStarted();
            FluentFuture.from(bootstrap.globalPlatform().submit(new CheckedFunction0<CompletableFuture<RspList<Object>>>() {
                @Override
                public CompletableFuture<RspList<Object>> apply() throws Throwable {
                    MethodCall call = new MethodCall(METHOD_ON_MODIFIED, line);
                    CompletableFuture<RspList<Object>> future = dispatcher.callRemoteMethodsWithFuture(null, call, RequestOptions.ASYNC());
                    stopWatch.stop();
                    return future;
                }
            })).addCallback(new FutureCallback<CompletableFuture<RspList<Object>>>() {
                @Override
                public void onSuccess(CompletableFuture<RspList<Object>> result) {
                    long time = stopWatch.getTime(TimeUnit.SECONDS);
                    if (time > 0) {
                        logger.error("notify operation took too long: {} for tables: {}", stopWatch, dependentTables);
                    }
                }
                @Override
                public void onFailure(Throwable t) {
                    logger.error(t.getMessage(), t);
                }
            }, MoreExecutors.directExecutor());
        }
    }
    @Override
    public void onTablesModify(String line) {
        Iterable<String> it = Splitter.on(',').omitEmptyStrings().split(line);
        ImmutableSet<String> tables = ImmutableSet.copyOf(it);
        ServerCacheNotification notification = new ServerCacheNotification(tables);
        notify.notify(notification);
    }
    @Override
    public void onCachePut(String cacheKey, byte[] id, byte[] value) {
        ReplicatedEbeanCache cache = (ReplicatedEbeanCache) caches.get(cacheKey);
        if (Objects.nonNull(cache)) {
            cache.onPut(id, value);
        }
    }
    @Override
    public void onCacheRemove(String cacheKey, byte[] id) {
        ReplicatedEbeanCache cache = (ReplicatedEbeanCache) caches.get(cacheKey);
        if (Objects.nonNull(cache)) {
            cache.onRemove(id);
        }
    }
    @Override
    public void onCacheClear(String cacheKey) {
        ServerCache cache = caches.get(cacheKey);
        if (Objects.nonNull(cache)) {
            if (cache instanceof ReplicatedEbeanCache) {
                ((ReplicatedEbeanCache) cache).onClear();
            } else if (cache instanceof JgroupsServerQueryCache) {
                ((JgroupsServerQueryCache) cache).onClear();
            }
        }
    }
    @Override
    public void onCacheClearAll(boolean preserveSimple) {
        for (ServerCache cache : caches.values()) {
            if (cache instanceof ReplicatedEbeanCache) {
                ((ReplicatedEbeanCache) cache).onClear();
            } else if (cache instanceof SimpleCache) {
                if (BooleanUtils.isFalse(preserveSimple)) {
                    ((SimpleCache) cache).onClear();
                }
            } else if (cache instanceof LocalEbeanCache) {
                ((LocalEbeanCache) cache).onClear();
            } else if (cache instanceof JgroupsServerQueryCache) {
                ((JgroupsServerQueryCache) cache).onClear();
            }
        }
    }
    @Override
    public ServerCache getCache(String cacheKey) {
        return caches.get(cacheKey);
    }
    @Override
    public void clearAllLocal() {
        for (ServerCache cache : caches.values()) {
            if (cache instanceof LocalEbeanCache) {
                ((LocalEbeanCache) cache).onClear();
            }
        }
    }
    @Override
    public void clearAllSimple() {
        for (ServerCache cache : caches.values()) {
            if (cache instanceof SimpleCache) {
                ((SimpleCache) cache).onClear();
            }
        }
    }
    @Override
    @SuppressWarnings("serial")
    public void clearAll(boolean preserveSimple) {
        onCacheClearAll(preserveSimple);

        StopWatch stopWatch = StopWatch.createStarted();
        FluentFuture.from(bootstrap.globalPlatform().submit(new CheckedFunction0<CompletableFuture<RspList<Object>>>() {
            @Override
            public CompletableFuture<RspList<Object>> apply() throws Throwable {
                MethodCall call = new MethodCall(METHOD_ON_CACHE_CLEAR_ALL, preserveSimple);
                CompletableFuture<RspList<Object>> future = dispatcher.callRemoteMethodsWithFuture(null, call, RequestOptions.ASYNC());
                stopWatch.stop();
                return future;
            }
        })).addCallback(new FutureCallback<CompletableFuture<RspList<Object>>>() {
            @Override
            public void onSuccess(CompletableFuture<RspList<Object>> result) {
                long time = stopWatch.getTime(TimeUnit.SECONDS);
                if (time > 0) {
                    logger.error("clearAll operation took too long: {}", stopWatch);
                }
            }
            @Override
            public void onFailure(Throwable t) {
                logger.error(t.getMessage(), t);
            }
        }, MoreExecutors.directExecutor());
    }
    @Override
    public SimpleCache createSimpleCache(String name) {
        ServerCache cache = caches.get(name);
        if (Objects.isNull(cache)) {
            cache = new SimpleCache(name);
            ServerCache prev = caches.putIfAbsent(name, cache);
            if (Objects.nonNull(prev)) {
                cache = prev;
            } else {
                new GuavaCacheMetrics<>(((SimpleCache) cache).get(), name, Collections.emptyList()).bindTo(bootstrap.meterRegistry());
            }
        }
        return (SimpleCache) cache;
    }
    @Override
    public ServerCache createCache(ServerCacheConfig config) {
        String cacheKey = config.getCacheKey();
        String shortName = config.getShortName();

        ServerCacheOptions cacheOptions = config.getCacheOptions();
        ServerCacheType cacheType = config.getType();
        CurrentTenantProvider tenantProvider = config.getTenantProvider();
        QueryCacheEntryValidate queryCacheValidate = config.getQueryCacheEntryValidate();

        ServerCache cache = caches.get(cacheKey);
        if (Objects.isNull(cache)) {
            Config prefixedView = bootstrap.cfg().getPrefixedView(cacheKey);
            Map<String, Object> configMap = new HashMap<>();
            for (String key : prefixedView.keys()) {
                Object rawProperty = prefixedView.getRawProperty(key);
                configMap.put(key, rawProperty);
            }

            //
            // ~ optional cache settings
            //
            int maxTtl = prefixedView.getInteger(EbeanCacheConfigurer.MAX_TTL, cacheOptions.getMaxSecsToLive());
            int maxIdle = prefixedView.getInteger(EbeanCacheConfigurer.MAX_IDLE, cacheOptions.getMaxIdleSecs());
            int maxSize = prefixedView.getInteger(EbeanCacheConfigurer.MAX_SIZE, cacheOptions.getMaxSize());

            int trimFrequency = (int) bootstrap.props().APP_TIMER_INTERVAL.get().toSeconds();

            ServerCacheOptions options = new ServerCacheOptions();
            options.setMaxSecsToLive(maxTtl);
            options.setMaxIdleSecs(maxIdle);
            options.setMaxSize(maxSize);
            options.setTrimFrequency(trimFrequency);

            ServerCacheConfig scc = new ServerCacheConfig(
                    cacheType,
                    cacheKey,
                    shortName,
                    options,
                    tenantProvider,
                    queryCacheValidate //
            );
            List<Tag> tags = Lists.newArrayList(
                    Tag.of("cacheType", cacheType.name().toLowerCase()),
                    Tag.of("shortName", shortName)//
            );

            if (config.isQueryCache()) {
                cache = new JgroupsServerQueryCache(dispatcher, new DefaultServerCacheConfig(scc));
                ((JgroupsServerQueryCache) cache).setBootstrap(bootstrap);
                ServerCache prev = caches.putIfAbsent(cacheKey, cache);
                if (Objects.nonNull(prev)) {
                    cache = prev;
                } else {
                    logger.debug("created query cache: {} using cfg {}",
                            cacheKey,
                            ToStringBuilder.reflectionToString(options, ToStringStyle.SHORT_PREFIX_STYLE));
                }
            } else {
                //
                // ~ fail fast in development mode and raise sentry error in production mode otherwise
                //
                if (BooleanUtils.isFalse(prefixedView.containsKey(EbeanCacheConfigurer.CACHE_MODE_LOCAL))) {
                    if (bootstrap.isProdMode()) {
                        logger.error("cache {} is not configured", cacheKey);
                    } else {
                        throw new IllegalStateException("cache %s is not configured".formatted(cacheKey));
                    }
                }

                boolean localMode = prefixedView.getBoolean(EbeanCacheConfigurer.CACHE_MODE_LOCAL, false);

                if (localMode) {
                    boolean neverExpire = bootstrap.props().CACHE_LOCAL_NEVER_EXPIRE.get();
                    if (neverExpire) {
                        cache = new LocalEbeanCache(cacheKey, maxSize, tenantProvider);
                        ServerCache prev = caches.putIfAbsent(cacheKey, cache);
                        if (Objects.nonNull(prev)) {
                            cache = prev;
                        } else {
                            new GuavaCacheMetrics<>(((LocalEbeanCache) cache).get(), cacheKey, tags).bindTo(bootstrap.meterRegistry());
                            logger.debug("created local cache: {} with max size: {}", cacheKey, maxSize);
                        }
                    } else {
                        cache = new LocalEbeanCache(cacheKey, new DefaultServerCacheConfig(scc), tenantProvider);
                        ServerCache prev = caches.putIfAbsent(cacheKey, cache);
                        if (Objects.nonNull(prev)) {
                            cache = prev;
                        } else {
                            new GuavaCacheMetrics<>(((LocalEbeanCache) cache).get(), cacheKey, tags).bindTo(bootstrap.meterRegistry());
                            logger.debug("created local cache: {} using cfg {}",
                                    cacheKey,
                                    ToStringBuilder.reflectionToString(options, ToStringStyle.NO_CLASS_NAME_STYLE));
                        }
                    }
                } else {
                    boolean neverExpire = bootstrap.props().CACHE_REPLICATED_NEVER_EXPIRE.get();
                    if (neverExpire) {
                        LocalCache local = new LocalEbeanCache(cacheKey, maxSize, tenantProvider);
                        cache = new ReplicatedEbeanCache(cacheKey, dispatcher, local, scc);
                        ((ReplicatedEbeanCache) cache).setBootstrap(bootstrap);
                        ServerCache prev = caches.putIfAbsent(cacheKey, cache);
                        if (Objects.nonNull(prev)) {
                            cache = prev;
                        } else {
                            new GuavaCacheMetrics<>(((LocalEbeanCache) local).get(), cacheKey, tags).bindTo(bootstrap.meterRegistry());
                            logger.debug("created replicated cache: {} with max size: {}", cacheKey, maxSize);
                        }
                    } else {
                        LocalCache local = new LocalEbeanCache(cacheKey, new DefaultServerCacheConfig(scc), tenantProvider);
                        cache = new ReplicatedEbeanCache(cacheKey, dispatcher, local, scc);
                        ((ReplicatedEbeanCache) cache).setBootstrap(bootstrap);
                        ServerCache prev = caches.putIfAbsent(cacheKey, cache);
                        if (Objects.nonNull(prev)) {
                            cache = prev;
                        } else {
                            new GuavaCacheMetrics<>(((LocalEbeanCache) local).get(), cacheKey, tags).bindTo(bootstrap.meterRegistry());
                            logger.debug("created replicated cache: {} using cfg {}",
                                    cacheKey,
                                    ToStringBuilder.reflectionToString(options, ToStringStyle.NO_CLASS_NAME_STYLE));
                        }
                    }
                }
            }

            //
            // ~ cleanUp
            //
            ServerCache tmp = Objects.requireNonNull(cache);
            executor.scheduleWithFixedDelay(new Runnable() {
                @Override
                public void run() {
                    if (tmp instanceof LocalEbeanCache) {
                        ((LocalEbeanCache) tmp).cleanUp();
                    } else if (tmp instanceof ReplicatedCache) {
                        ((ReplicatedCache) tmp).cleanUp();
                    } else if (tmp instanceof SimpleCache) {
                        ((SimpleCache) tmp).cleanUp();
                    } else if (tmp instanceof DefaultServerQueryCache) {
                        ((DefaultServerQueryCache) tmp).runEviction();
                    }
                }
            }, trimFrequency, trimFrequency, TimeUnit.SECONDS);
        }

        return cache;
    }
}
