package com.turbospaces.ebean;

import java.math.BigDecimal;
import java.time.Duration;
import java.util.Collections;
import java.util.Set;

import javax.inject.Inject;

import org.awaitility.Awaitility;
import org.awaitility.core.ThrowingRunnable;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Configurable;
import org.springframework.cloud.service.ServiceInfo;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.annotation.Bean;

import com.google.common.collect.Sets;
import com.turbospaces.boot.AbstractBootstrapAware;
import com.turbospaces.boot.MockCloud;
import com.turbospaces.boot.SimpleBootstrap;
import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.ebean.query.QAccount;
import com.turbospaces.jdbc.HikariDataSourceFactoryBean;
import com.turbospaces.jgroups.JGroupsFactoryBean;
import com.turbospaces.plugins.FlywayBootstrapInitializer;
import com.turbospaces.ups.H2ServiceInfo;
import com.turbospaces.ups.UPSs;

import io.ebean.DuplicateKeyException;
import io.ebean.Transaction;
import io.ebean.cache.ServerCache;
import io.ebean.cache.ServerCacheType;
import io.ebean.config.dbplatform.PlatformIdGenerator;

public class EbeanInfinispanPluginTest {
    @Test
    public void works() throws Throwable {
        MockCloud builder = MockCloud.newMock();
        ApplicationConfig cfg = builder.build();
        ApplicationProperties props = new ApplicationProperties(cfg);
        SimpleBootstrap bootstrap = new SimpleBootstrap(props, AppConfig.class);
        bootstrap.withH2(true, bootstrap.spaceName());
        H2ServiceInfo ownerUps = UPSs.findRequiredServiceInfoByName(bootstrap, UPSs.H2_OWNER);

        bootstrap.addBootstrapRegistryInitializer(new FlywayBootstrapInitializer(ownerUps, "CORE"));
        ConfigurableApplicationContext applicationContext = bootstrap.run();

        try {
            EbeanJpaManager ebean = (EbeanJpaManager) applicationContext.getBean(JpaManager.class);
            Set<Long> set = Sets.newHashSet();
            long now = System.currentTimeMillis();

            try (Transaction tx = ebean.beginTransaction()) {
                for (int i = 0; i < 16 * 1024; i++) {
                    PlatformIdGenerator generator = ebean.idGenerator(AccountSeq.class);
                    Account account = new Account();

                    long nextId = (long) generator.nextId(tx);
                    Assertions.assertTrue(set.add(nextId));

                    account.setId(now + nextId);
                    account.setFraud(new FraudJson(Collections.emptyMap()));
                    account.setUsername("username_" + account.getId());
                    account.setFirstName("f_" + account.getId());
                    account.setLastName("l_" + account.getId());
                    account.setDetails(Collections.emptyMap());

                    ebean.save(account, tx);

                }

                tx.commit();
            }

            long min = set.stream().min(Long::compare).get();
            long max = set.stream().max(Long::compare).get();

            Assertions.assertEquals(min, 1);
            Assertions.assertEquals(max, 16 * 1024);
            Assertions.assertEquals(ebean.idGenerators().size(), 1);

        } finally {
            bootstrap.shutdown();
        }
    }

    private static class RollbackBean extends AbstractBootstrapAware implements InitializingBean {
        private final JpaManager ebean;
        private final CacheManager manager;

        @Inject
        public RollbackBean(JpaManager ebean, CacheManager manager) {
            this.ebean = ebean;
            this.manager = manager;
        }
        @Override
        public void afterPropertiesSet() throws Exception {
            String tname = UTMTemplate.class.getName();
            long id = Math.abs(hashCode());

            Account account;
            try (Transaction tx = ebean.newTransaction("cache-bean-save")) {
                account = new Account();

                account.setId(id);
                account.setFraud(new FraudJson(Collections.emptyMap()));
                account.setUsername("username_" + account.getId());
                account.setFirstName("f_" + account.getId());
                account.setLastName("l_" + account.getId());
                account.setDetails(Collections.emptyMap());

                ebean.save(account);
                tx.flush();

                UTMTemplate template1 = new UTMTemplate(account, "utm-1");
                ebean.save(template1, tx);

                UTMTemplate template2 = new UTMTemplate(account, "utm-2");
                ebean.save(template2, tx);

                ebean.createQuery(UTMTemplate.class).usingTransaction(tx).where().eq("account", account).eq("campaign", template1.getCampaign()).findOne();
                ebean.createQuery(UTMTemplate.class).usingTransaction(tx).where().eq("account", account).eq("campaign", template2.getCampaign()).findOne();

                UTMTemplate template3 = new UTMTemplate(account, "utm-2");
                ebean.save(template3, tx);

                tx.rollback();
            } catch (DuplicateKeyException err) {
                logger.warn(err.getMessage(), err);
            }

            ServerCache tCache = manager.getCache(tname + ServerCacheType.BEAN.code());

            Assertions.assertEquals(2, tCache.size());
        }
    }

    public static class CacheBean extends AbstractBootstrapAware implements InitializingBean {
        private final JpaManager ebean;
        private final CacheManager manager;

        @Inject
        public CacheBean(JpaManager ebean, CacheManager manager) {
            this.ebean = ebean;
            this.manager = manager;
        }
        @Override
        public void afterPropertiesSet() throws Exception {
            String aname = Account.class.getName();
            String abname = AccountBalance.class.getName();

            long id = Math.abs(hashCode());
            Account account;
            try (Transaction tx = ebean.newTransaction("cache-bean-save")) {
                account = new Account();

                account.setId(id);
                account.setFraud(new FraudJson(Collections.emptyMap()));
                account.setUsername("username_" + account.getId());
                account.setFirstName("f_" + account.getId());
                account.setLastName("l_" + account.getId());
                account.setDetails(Collections.emptyMap());

                AccountBalance balance1 = new AccountBalance(account, "USD");
                balance1.setAmount(BigDecimal.TEN);
                account.getBalances().add(balance1);

                AccountBalance balance2 = new AccountBalance(account, "EUR");
                balance2.setAmount(BigDecimal.ONE);
                account.getBalances().add(balance2);

                AccountBalance balance3 = new AccountBalance(account, "UAH");
                balance3.setAmount(BigDecimal.ONE);
                account.getBalances().add(balance3);

                ebean.save(account);
                tx.commit();
            }

            try (Transaction tx = ebean.newTransaction("cache-bean-save-all")) {
                for (Account it : ebean.find(Account.class).findList()) {
                    it.setAge(it.getAge() + 1);
                    it.setGameplayInfo(new GameplayInfo());
                    ebean.save(it, tx);
                }

                tx.commit();
            }

            for (int i = 0; i < 10; i++) {
                logger.debug("it ::: {}", i);
                try (Transaction tx = ebean.newReadOnlyTransaction("cache-bean-readonly")) {
                    tx.setReadOnly(true);
                    account = ebean.find(Account.class, id);
                    account.getBalances().size(); // id cache
                    for (AccountBalance balance : account.getBalances()) {
                        balance.getAmount().toString(); // balance cache
                    }
                }
            }

            for (int i = 0; i < 10; i++) {
                try (Transaction tx = ebean.newReadOnlyTransaction("cache-bean-readonly")) {
                    tx.setReadOnly(true);
                    Set<Account> accounts = ebean.createQuery(Account.class).setUseQueryCache(true).findSet();
                    accounts.size();
                }
            }
            // Assert.assertEquals( 1, qCache.size() );

            try (Transaction tx = ebean.newTransaction("cache-bean-save")) {
                account = ebean.find(Account.class, id);
                account.setAge(32);
                ebean.save(account);
                tx.commit();
            }
            Awaitility.await().atMost(Duration.ofSeconds(10)).untilAsserted(new ThrowingRunnable() {
                @Override
                public void run() throws Throwable {
                    // Assert.assertEquals( 0, qCache.size() );
                }
            });

            // Cache<Object, Object> qCache = channel.getCache( aname + ServerCacheType.QUERY.code() );
            ServerCache aCache = manager.getCache(aname + ServerCacheType.BEAN.code());
            ServerCache bCache = manager.getCache(abname + ServerCacheType.BEAN.code());
            ServerCache cCache = manager.getCache(aname + "." + QAccount.alias().balances.toString() + ServerCacheType.COLLECTION_IDS.code());

            Assertions.assertEquals(1, aCache.size());
            Assertions.assertEquals(2, bCache.size());
            Assertions.assertEquals(1, cCache.size());

            ebean.cacheManager().beanCache(Account.class).clear();
            Assertions.assertEquals(0, aCache.size());
            Assertions.assertEquals(2, bCache.size()); // ~ ebean is not clearing children
            Assertions.assertEquals(1, cCache.size());

            // clear bean cache
            ebean.cacheManager().beanCache(AccountBalance.class).clear();
            Assertions.assertEquals(0, bCache.size());

            ebean.cacheManager().clear(Account.class);
            Assertions.assertEquals(0, cCache.size());
        }
    }

    @Configurable
    public static class AppConfig extends AbstractBootstrapAware {
        @Bean
        public HikariDataSourceFactoryBean ds() {
            ServiceInfo appUps = UPSs.findRequiredServiceInfoByName(bootstrap, UPSs.H2_APP);
            return new HikariDataSourceFactoryBean(appUps);
        }
        @Bean
        public JGroupsFactoryBean jgroups() {
            return new JGroupsFactoryBean();
        }
        @Bean
        public JGroupCacheManagerFactoryBean cacheManager(JGroupsFactoryBean factory) throws Exception {
            return new JGroupCacheManagerFactoryBean(factory.getObject());
        }
        @Bean
        public EbeanDatabaseConfig ebeanConfig(HikariDataSourceFactoryBean factory) throws Exception {
            EbeanDatabaseConfig cfg = new EbeanDatabaseConfig(factory.getObject(), bootstrap.props());

            cfg.addClass(AccountSeq.class);
            cfg.addClass(Account.class);
            cfg.addClass(GameplayInfo.class);
            cfg.addClass(FraudJson.class);
            cfg.addClass(AccountBalance.class);
            cfg.addClass(AccountBalanceId.class);
            cfg.addClass(AccountBalanceSnapshot.class);
            cfg.addClass(AccountBalanceSnapshotId.class);
            cfg.addClass(UTMTemplate.class);

            cfg.setMaxSize(Account.class, 1);
            cfg.setMaxSizeQuery(Account.class, 1);

            cfg.setMaxSize(Account.class, QAccount.alias().balances, 1);
            cfg.setMaxSize(UTMTemplate.class, Byte.SIZE);
            cfg.setMaxSize(AccountBalance.class, 2);
            cfg.setMaxSize(AccountBalanceSnapshot.class, 4);
            cfg.setMaxSize(Account.class, QAccount.alias().utmTemplates, Integer.MAX_VALUE);

            return cfg;
        }
        @Bean
        public EbeanFactoryBean ebean(EbeanDatabaseConfig config, JGroupCacheManagerFactoryBean cache) throws Exception {
            return new EbeanFactoryBean(config, cache.getObject());
        }
        @Bean
        public RollbackBean rollbackBean(EbeanFactoryBean ebean, CacheManager manager) throws Exception {
            return new RollbackBean(ebean.getObject(), manager);
        }
        @Bean
        public CacheBean cacheBean(EbeanFactoryBean ebean, CacheManager manager) throws Exception {
            return new CacheBean(ebean.getObject(), manager);
        }
    }
}
