package com.turbospaces.ebean;

import java.lang.reflect.Method;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.jgroups.blocks.MethodCall;
import org.jgroups.blocks.MethodLookup;
import org.jgroups.blocks.RequestOptions;
import org.jgroups.blocks.RpcDispatcher;
import org.jgroups.util.RspList;

import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.netflix.archaius.api.Config;
import com.turbospaces.boot.AbstractBootstrapAware;

import io.ebean.BackgroundExecutor;
import io.ebean.cache.QueryCacheEntryValidate;
import io.ebean.cache.ServerCache;
import io.ebean.cache.ServerCacheConfig;
import io.ebean.cache.ServerCacheFactory;
import io.ebean.cache.ServerCacheNotification;
import io.ebean.cache.ServerCacheNotify;
import io.ebean.cache.ServerCacheOptions;
import io.ebean.cache.ServerCacheType;
import io.ebean.config.CurrentTenantProvider;
import io.ebean.config.DatabaseConfig;
import io.ebeaninternal.server.cache.DefaultServerCacheConfig;
import io.ebeaninternal.server.cache.DefaultServerQueryCache;
import io.micrometer.core.instrument.Tag;
import io.micrometer.core.instrument.binder.cache.GuavaCacheMetrics;

public class JGroupsCacheManager extends AbstractBootstrapAware implements MethodLookup, CacheManager, ServerCacheFactory, ServerCacheNotify {
    public static short METHOD_ON_CACHE_PUT = 1;
    public static short METHOD_ON_CHANGE_REMOVE = 2;
    public static short METHOD_ON_CACHE_CLEAR = 3;
    public static short METHOD_ON_MODIFIED = 4;
    public static short METHOD_ON_CACHE_CLEAR_ALL = 5;
    public static ImmutableMap<Short, Method> METHODS;

    static {
        try {
            ImmutableMap.Builder<Short, Method> b = ImmutableMap.builder();
            b.put(METHOD_ON_MODIFIED, JGroupsCacheManager.class.getMethod("onTablesModify", String.class));
            b.put(METHOD_ON_CACHE_PUT, JGroupsCacheManager.class.getMethod("onCachePut", String.class, byte[].class, byte[].class));
            b.put(METHOD_ON_CHANGE_REMOVE, JGroupsCacheManager.class.getMethod("onCacheRemove", String.class, byte[].class));
            b.put(METHOD_ON_CACHE_CLEAR, JGroupsCacheManager.class.getMethod("onCacheClear", String.class));
            b.put(METHOD_ON_CACHE_CLEAR_ALL, JGroupsCacheManager.class.getMethod("onCacheClearAll", boolean.class));

            METHODS = b.build();
        } catch (NoSuchMethodException err) {
            throw new Error(err);
        }
    }

    private final RpcDispatcher dispatcher;
    private final RequestOptions requestOptions;
    private final ConcurrentMap<String, ServerCache> caches;
    private BackgroundExecutor executor;
    private ServerCacheNotify notify;

    public JGroupsCacheManager(RpcDispatcher dispatcher, RequestOptions requestOptions) {
        this.dispatcher = Objects.requireNonNull(dispatcher);
        this.requestOptions = Objects.requireNonNull(requestOptions);
        this.caches = Maps.newConcurrentMap();
    }
    @Override
    public Method findMethod(short id) {
        return METHODS.get(id);
    }
    @Override
    public ServerCacheFactory create(DatabaseConfig config, BackgroundExecutor backgroundExecutor) {
        this.executor = Objects.requireNonNull(backgroundExecutor);
        return this;
    }
    @Override
    public ServerCacheNotify createCacheNotify(ServerCacheNotify cacheNotify) {
        this.notify = Objects.requireNonNull(cacheNotify);
        return this;
    }
    @Override
    public void notify(ServerCacheNotification notification) {
        if (notification.getDependentTables() != null && !notification.getDependentTables().isEmpty()) {
            Set<String> dependentTables = notification.getDependentTables();
            String line = Joiner.on(',').join(dependentTables);
            logger.debug("Publish TableMods - {}", line);

            try {
                MethodCall call = new MethodCall(METHOD_ON_MODIFIED, line);
                RspList<Object> l = dispatcher.callRemoteMethods(null, call, requestOptions);
                if (Objects.nonNull(l)) {
                    ReplicatedCache.trace(logger, l);
                }
            } catch (Throwable t) {
                throw new RuntimeException(t);
            }
        }
    }
    @Override
    public void onTablesModify(String line) {
        Iterable<String> it = Splitter.on(',').omitEmptyStrings().split(line);
        ImmutableSet<String> tables = ImmutableSet.copyOf(it);
        ServerCacheNotification notification = new ServerCacheNotification(tables);
        notify.notify(notification);
    }
    @Override
    public void onCachePut(String cacheKey, byte[] id, byte[] value) {
        ReplicatedEbeanCache cache = (ReplicatedEbeanCache) caches.get(cacheKey);
        if (cache != null) {
            cache.onPut(id, value);
        }
    }
    @Override
    public void onCacheRemove(String cacheKey, byte[] id) {
        ReplicatedEbeanCache cache = (ReplicatedEbeanCache) caches.get(cacheKey);
        if (cache != null) {
            cache.onRemove(id);
        }
    }
    @Override
    public void onCacheClear(String cacheKey) {
        ServerCache cache = caches.get(cacheKey);
        if (cache != null) {
            if (cache instanceof ReplicatedEbeanCache) {
                ((ReplicatedEbeanCache) cache).onClear(); // ~ only for replicated scenarios
            }
        }
    }
    @Override
    public void onCacheClearAll(boolean preserveSimple) {
        for (ServerCache cache : caches.values()) {
            if (cache instanceof ReplicatedEbeanCache) {
                ((ReplicatedEbeanCache) cache).onClear();
            } else if (cache instanceof SimpleCache) {
                if (BooleanUtils.isFalse(preserveSimple)) {
                    ((SimpleCache) cache).onClear();
                }
            } else if (cache instanceof LocalEbeanCache) {
                ((LocalEbeanCache) cache).onClear();
            }
        }
    }
    @Override
    public ServerCache getCache(String cacheKey) {
        return caches.get(cacheKey);
    }
    @Override
    public void clearAllLocal() {
        for (ServerCache cache : caches.values()) {
            if (cache instanceof ReplicatedEbeanCache) {} // ~ ignore replicated
            else if (cache instanceof SimpleCache) {} // ~ ignore simple
            else if (cache instanceof LocalEbeanCache) {
                ((LocalEbeanCache) cache).onClear();
            }
        }
    }
    @Override
    public void clearAllSimple() {
        for (ServerCache cache : caches.values()) {
            if (cache instanceof ReplicatedEbeanCache) {} // ~ ignore replicated
            else if (cache instanceof LocalEbeanCache) {} // ~ ignore local
            else if (cache instanceof SimpleCache) {
                ((SimpleCache) cache).onClear();
            }
        }
    }
    @Override
    public void clearAll(boolean preserveSimple) {
        try {
            MethodCall call = new MethodCall(METHOD_ON_CACHE_CLEAR_ALL, preserveSimple);
            RspList<Object> l = dispatcher.callRemoteMethods(null, call, requestOptions);
            if (Objects.nonNull(l)) {
                ReplicatedCache.trace(logger, l);
            }
        } catch (Throwable t) {
            throw new RuntimeException(t);
        } finally {
            onCacheClearAll(preserveSimple);
        }
    }
    @Override
    public SimpleCache createSimpleCache(String name) {
        ServerCache cache = caches.get(name);
        if (Objects.isNull(cache)) {
            cache = new SimpleCache(name);
            ServerCache prev = caches.putIfAbsent(name, cache);
            if (Objects.nonNull(prev)) {
                cache = prev;
            } else {
                new GuavaCacheMetrics<>(((SimpleCache) cache).get(), name, Collections.emptyList()).bindTo(bootstrap.meterRegistry());
            }
        }
        return (SimpleCache) cache;
    }
    @Override
    public ServerCache createCache(ServerCacheConfig config) {
        String cacheKey = config.getCacheKey();
        String shortName = config.getShortName();

        ServerCacheOptions cacheOptions = config.getCacheOptions();
        ServerCacheType cacheType = config.getType();
        CurrentTenantProvider tenantProvider = config.getTenantProvider();
        QueryCacheEntryValidate queryCacheValidate = config.getQueryCacheEntryValidate();

        ServerCache cache = caches.get(cacheKey);
        if (Objects.isNull(cache)) {
            Config prefixedView = bootstrap.cfg().getPrefixedView(cacheKey);
            if (prefixedView.isEmpty()) {
                throw new IllegalArgumentException(String.format("cache: %s is not configured", cacheKey));
            }
            Iterator<String> it = prefixedView.getKeys();
            Map<String, Object> configMap = new HashMap<>();
            while (it.hasNext()) {
                String key = it.next();
                Object rawProperty = prefixedView.getRawProperty(key);
                configMap.put(key, rawProperty);
            }

            //
            // ~ optional cache settings
            //
            int maxTtl = prefixedView.getInteger(EbeanCacheConfigurer.MAX_TTL, cacheOptions.getMaxSecsToLive());
            int maxIdle = prefixedView.getInteger(EbeanCacheConfigurer.MAX_IDLE, cacheOptions.getMaxIdleSecs());
            int maxSize = prefixedView.getInteger(EbeanCacheConfigurer.MAX_SIZE, cacheOptions.getMaxSize());

            int trimFrequency = (int) bootstrap.props().APP_TIMER_INTERVAL.get().getSeconds();

            ServerCacheOptions options = new ServerCacheOptions();
            options.setMaxSecsToLive(maxTtl);
            options.setMaxIdleSecs(maxIdle);
            options.setMaxSize(maxSize);
            options.setTrimFrequency(trimFrequency);

            ServerCacheConfig scc = new ServerCacheConfig(
                    cacheType,
                    cacheKey,
                    shortName,
                    options,
                    tenantProvider,
                    queryCacheValidate //
            );
            List<Tag> tags = Lists.newArrayList(
                    Tag.of("cacheType", cacheType.name().toLowerCase()),
                    Tag.of("shortName", shortName)//
            );

            if (config.isQueryCache()) {
                cache = new DefaultServerQueryCache(new DefaultServerCacheConfig(scc));
                ServerCache prev = caches.putIfAbsent(cacheKey, cache);
                if (Objects.nonNull(prev)) {
                    cache = prev;
                } else {
                    logger.debug("created query cache: {} using cfg {}",
                            cacheKey,
                            ToStringBuilder.reflectionToString(options, ToStringStyle.SHORT_PREFIX_STYLE) //
                    );
                }
            } else {
                boolean localMode = prefixedView.getBoolean(EbeanCacheConfigurer.CACHE_MODE_LOCAL, true);

                if (localMode) {
                    boolean neverExpire = bootstrap.props().CACHE_LOCAL_NEVER_EXPIRE.get();
                    if (neverExpire) {
                        cache = new LocalEbeanCache(cacheKey, maxSize, tenantProvider);
                        ServerCache prev = caches.putIfAbsent(cacheKey, cache);
                        if (Objects.nonNull(prev)) {
                            cache = prev;
                        } else {
                            new GuavaCacheMetrics<>(((LocalEbeanCache) cache).get(), cacheKey, tags).bindTo(bootstrap.meterRegistry());
                            logger.debug("created local cache: {} with max size: {}", cacheKey, maxSize);
                        }
                    } else {
                        cache = new LocalEbeanCache(cacheKey, new DefaultServerCacheConfig(scc), tenantProvider);
                        ServerCache prev = caches.putIfAbsent(cacheKey, cache);
                        if (Objects.nonNull(prev)) {
                            cache = prev;
                        } else {
                            new GuavaCacheMetrics<>(((LocalEbeanCache) cache).get(), cacheKey, tags).bindTo(bootstrap.meterRegistry());
                            logger.debug("created local cache: {} using cfg {}",
                                    cacheKey,
                                    ToStringBuilder.reflectionToString(options, ToStringStyle.NO_CLASS_NAME_STYLE) //
                            );
                        }
                    }
                } else {
                    boolean neverExpire = bootstrap.props().CACHE_REPLICATED_NEVER_EXPIRE.get();
                    if (neverExpire) {
                        LocalCache local = new LocalEbeanCache(cacheKey, maxSize, tenantProvider);
                        cache = new ReplicatedEbeanCache(cacheKey, dispatcher, requestOptions, local, scc);
                        ServerCache prev = caches.putIfAbsent(cacheKey, cache);
                        if (Objects.nonNull(prev)) {
                            cache = prev;
                        } else {
                            new GuavaCacheMetrics<>(((LocalEbeanCache) local).get(), cacheKey, tags).bindTo(bootstrap.meterRegistry());
                            logger.debug("created replicated cache: {} with max size: {}", cacheKey, maxSize);
                        }
                    } else {
                        LocalCache local = new LocalEbeanCache(cacheKey, new DefaultServerCacheConfig(scc), tenantProvider);
                        cache = new ReplicatedEbeanCache(cacheKey, dispatcher, requestOptions, local, scc);
                        ServerCache prev = caches.putIfAbsent(cacheKey, cache);
                        if (Objects.nonNull(prev)) {
                            cache = prev;
                        } else {
                            new GuavaCacheMetrics<>(((LocalEbeanCache) local).get(), cacheKey, tags).bindTo(bootstrap.meterRegistry());
                            logger.debug("created replicated cache: {} using cfg {}",
                                    cacheKey,
                                    ToStringBuilder.reflectionToString(options, ToStringStyle.NO_CLASS_NAME_STYLE) //
                            );
                        }
                    }
                }
            }

            // ~ cleanUp
            ServerCache tmp = Objects.requireNonNull(cache);
            executor.scheduleWithFixedDelay(new Runnable() {
                @Override
                public void run() {
                    if (tmp instanceof LocalEbeanCache) {
                        ((LocalEbeanCache) tmp).get().cleanUp();
                    } else if (tmp instanceof DefaultServerQueryCache) {
                        ((DefaultServerQueryCache) tmp).runEviction();
                    }
                }
            }, trimFrequency, trimFrequency, TimeUnit.SECONDS);
        }

        return cache;
    }
}
