package com.turbospaces.ebean;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.ZoneOffset;

import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Configurable;
import org.springframework.cloud.service.ServiceInfo;
import org.springframework.context.annotation.Bean;

import com.turbospaces.boot.AbstractBootstrapAware;
import com.turbospaces.boot.MockCloud;
import com.turbospaces.boot.SimpleBootstrap;
import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.ebean.query.QAccount;
import com.turbospaces.ebean.query.QAccountBalance;
import com.turbospaces.jdbc.HikariDataSourceFactoryBean;
import com.turbospaces.jgroups.JGroupsFactoryBean;
import com.turbospaces.plugins.FlywayBootstrapInitializer;
import com.turbospaces.ups.UPSs;

import io.ebean.Database;

public class CompositeKeyTest {
    @Test
    public void works() throws Throwable {
        MockCloud builder = MockCloud.newMock();
        ApplicationConfig cfg = builder.build();
        ApplicationProperties props = new ApplicationProperties(cfg);
        EbeanCacheConfigurer mngr = new DefaultEbeanCacheConfigurer(props);

        //
        // ~ local cache mode
        //
        mngr.setMaxSize(Account.class, 1);
        mngr.setMaxSize(Account.class, QAccount.alias().balances, 1);

        //
        // ~ replicated cache mode
        //
        mngr.setMaxSize(UTMTemplate.class, Byte.SIZE);
        mngr.setMaxSize(AccountBalance.class, 10);
        mngr.setMaxSize(AccountBalance.class, QAccountBalance.alias().snapshots, 10);
        mngr.setMaxSize(AccountBalanceSnapshot.class, 30);
        mngr.setMaxSize(Account.class, QAccount.alias().utmTemplates, Integer.MAX_VALUE);

        //
        // ~ query cache mode
        //
        mngr.setMaxSizeQuery(Account.class, 1);

        SimpleBootstrap bootstrap = new SimpleBootstrap(new ApplicationProperties(cfg), AppConfig.class);
        bootstrap.withH2(true, bootstrap.spaceName());

        ServiceInfo ownerUps = UPSs.findRequiredServiceInfoByName(bootstrap, UPSs.H2_OWNER);

        bootstrap.addBootstrapRegistryInitializer(new FlywayBootstrapInitializer(ownerUps, "CORE"));
        bootstrap.run();

        Database ebean = bootstrap.context().getBean(Database.class);

        Account account = new Account();
        account.setId(System.currentTimeMillis());
        account.setUsername("username_" + account.getId());
        account.setFirstName("f_" + account.getId());
        account.setLastName("l_" + account.getId());
        account.setAge(18);
        account.setBirthDate(LocalDate.now(ZoneOffset.UTC).minusYears(account.getAge()));
        ebean.save(account);

        AccountBalance balance = new AccountBalance(account, "USD");
        balance.setAmount(BigDecimal.ONE);

        AccountBalanceSnapshot snapshot1 = new AccountBalanceSnapshot(balance, LocalDate.now(ZoneOffset.UTC));
        AccountBalanceSnapshot snapshot2 = new AccountBalanceSnapshot(balance, LocalDate.now(ZoneOffset.UTC).plusDays(1));

        balance.getSnapshots().add(snapshot1);
        balance.getSnapshots().add(snapshot2);
        ebean.save(balance);

        QAccountBalance q = new QAccountBalance(ebean);
        q.setId(balance.getPk());
        q.findList();

        try {

        } finally {
            bootstrap.shutdown();
        }
    }

    @Configurable
    public static class AppConfig extends AbstractBootstrapAware {
        @Bean
        public HikariDataSourceFactoryBean ds() {
            ServiceInfo appUps = UPSs.findRequiredServiceInfoByName(bootstrap, UPSs.H2_APP);
            return new HikariDataSourceFactoryBean(appUps);
        }
        @Bean
        public JGroupsFactoryBean jgroups() {
            return new JGroupsFactoryBean();
        }
        @Bean
        public JGroupCacheManagerFactoryBean cacheManager(JGroupsFactoryBean factory) throws Exception {
            return new JGroupCacheManagerFactoryBean(factory.getObject());
        }
        @Bean
        public EbeanDatabaseConfig ebeanConfig(HikariDataSourceFactoryBean factory) throws Exception {
            EbeanDatabaseConfig cfg = new EbeanDatabaseConfig(factory.getObject(), bootstrap.props());

            cfg.addClass(Account.class);
            cfg.addClass(GameplayInfo.class);
            cfg.addClass(FraudJson.class);
            cfg.addClass(UTMTemplate.class);
            cfg.addClass(AccountBalance.class);
            cfg.addClass(AccountBalanceId.class);
            cfg.addClass(AccountBalanceSnapshot.class);
            cfg.addClass(AccountBalanceSnapshotId.class);

            return cfg;
        }
        @Bean
        public EbeanFactoryBean ebean(EbeanDatabaseConfig config, JGroupCacheManagerFactoryBean cache) throws Exception {
            return new EbeanFactoryBean(config, cache.getObject());
        }
    }
}
