package com.turbospaces.ebean;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;

import com.netflix.archaius.api.Config;
import com.turbospaces.boot.AbstractBootstrapAware;

import io.ebean.BackgroundExecutor;
import io.ebean.cache.QueryCacheEntryValidate;
import io.ebean.cache.ServerCache;
import io.ebean.cache.ServerCacheConfig;
import io.ebean.cache.ServerCacheFactory;
import io.ebean.cache.ServerCacheNotification;
import io.ebean.cache.ServerCacheNotify;
import io.ebean.cache.ServerCacheOptions;
import io.ebean.cache.ServerCacheType;
import io.ebean.config.CurrentTenantProvider;
import io.ebean.config.DatabaseConfig;
import io.ebeaninternal.server.cache.DefaultServerCacheConfig;
import io.ebeaninternal.server.cache.DefaultServerQueryCache;

public class MockCacheManager extends AbstractBootstrapAware implements CacheManager, ServerCacheFactory {
    private final Map<String, ServerCache> caches = new ConcurrentHashMap<>();
    private BackgroundExecutor executor;

    @Override
    public ServerCacheFactory create(DatabaseConfig config, BackgroundExecutor backgroupExecutor) {
        this.executor = Objects.requireNonNull(backgroupExecutor);
        return this;
    }
    @Override
    public ServerCacheNotify createCacheNotify(ServerCacheNotify listener) {
        return new ServerCacheNotify() {
            @Override
            public void notify(ServerCacheNotification notification) {

            }
        };
    }
    @Override
    public SimpleCache createSimpleCache(String name) {
        ServerCache cache = caches.get(name);
        if (Objects.isNull(cache)) {
            cache = new SimpleCache(name);
            ServerCache prev = caches.putIfAbsent(name, cache);
            if (Objects.nonNull(prev)) {
                cache = prev;
            }
        }
        return (SimpleCache) cache;
    }
    @Override
    public ServerCache createCache(ServerCacheConfig config) {
        String cacheKey = config.getCacheKey();
        String shortName = config.getShortName();

        ServerCacheOptions cacheOptions = config.getCacheOptions();
        ServerCacheType cacheType = config.getType();
        CurrentTenantProvider tenantProvider = config.getTenantProvider();
        QueryCacheEntryValidate queryCacheValidate = config.getQueryCacheEntryValidate();

        ServerCache cache = caches.get(cacheKey);
        if (Objects.isNull(cache)) {
            Config prefixedView = bootstrap.cfg().getPrefixedView(cacheKey);
            Iterator<String> it = prefixedView.getKeys();
            Map<String, Object> configMap = new HashMap<>();
            while (it.hasNext()) {
                String key = it.next();
                Object rawProperty = prefixedView.getRawProperty(key);
                configMap.put(key, rawProperty);
            }

            int maxTtl = prefixedView.getInteger(EbeanCacheConfigurer.MAX_TTL, cacheOptions.getMaxSecsToLive());
            int maxIdle = prefixedView.getInteger(EbeanCacheConfigurer.MAX_IDLE, cacheOptions.getMaxIdleSecs());
            int maxSize = prefixedView.getInteger(EbeanCacheConfigurer.MAX_SIZE, cacheOptions.getMaxSize());
            int trimFrequency = (int) bootstrap.props().APP_TIMER_INTERVAL.get().getSeconds();

            ServerCacheOptions options = new ServerCacheOptions();
            options.setMaxSecsToLive(maxTtl);
            options.setMaxIdleSecs(maxIdle);
            options.setMaxSize(maxSize);
            options.setTrimFrequency(trimFrequency);

            ServerCacheConfig scc = new ServerCacheConfig(
                    cacheType,
                    cacheKey,
                    shortName,
                    options,
                    tenantProvider,
                    queryCacheValidate //
            );

            if (config.isQueryCache()) {
                cache = new DefaultServerQueryCache(new DefaultServerCacheConfig(scc));
                ServerCache prev = caches.putIfAbsent(cacheKey, cache);
                if (Objects.nonNull(prev)) {
                    cache = prev;
                } else {
                    logger.debug("created query cache: {} using cfg {}",
                            cacheKey,
                            ToStringBuilder.reflectionToString(options, ToStringStyle.SHORT_PREFIX_STYLE) //
                    );
                }
            } else {
                boolean neverExpire = bootstrap.props().CACHE_LOCAL_NEVER_EXPIRE.get();
                if (neverExpire) {
                    cache = new LocalEbeanCache(cacheKey, maxSize, tenantProvider);
                    ServerCache prev = caches.putIfAbsent(cacheKey, cache);
                    if (Objects.nonNull(prev)) {
                        cache = prev;
                    } else {
                        logger.debug("created local cache: {} with max size: {}", cacheKey, maxSize);
                    }
                } else {
                    cache = new LocalEbeanCache(cacheKey, new DefaultServerCacheConfig(scc), tenantProvider);
                    ServerCache prev = caches.putIfAbsent(cacheKey, cache);
                    if (Objects.nonNull(prev)) {
                        cache = prev;
                    } else {
                        logger.debug("created local cache: {} using cfg {}",
                                cacheKey,
                                ToStringBuilder.reflectionToString(options, ToStringStyle.NO_CLASS_NAME_STYLE) //
                        );
                    }
                }
            }

            // ~ cleanUp
            ServerCache tmp = Objects.requireNonNull(cache);
            executor.scheduleWithFixedDelay(new Runnable() {
                @Override
                public void run() {
                    if (tmp instanceof LocalEbeanCache) {
                        ((LocalEbeanCache) tmp).get().cleanUp();
                    } else if (tmp instanceof DefaultServerQueryCache) {
                        ((DefaultServerQueryCache) tmp).runEviction();
                    }
                }
            }, trimFrequency, trimFrequency, TimeUnit.SECONDS);
        }

        return cache;
    }
    @Override
    public ServerCache getCache(String string) {
        return caches.get(string);
    }
    @Override
    public void clearAllLocal() {
        for (ServerCache cache : caches.values()) {
            if (cache instanceof SimpleCache) {} // ~ ignore simple
            else if (cache instanceof LocalEbeanCache) {
                ((LocalEbeanCache) cache).onClear();
            }
        }
    }
    @Override
    public void clearAllSimple() {
        for (ServerCache cache : caches.values()) {
            if (cache instanceof LocalEbeanCache) {} // ~ ignore local
            else if (cache instanceof SimpleCache) {
                ((SimpleCache) cache).onClear();
            }
        }
    }
    @Override
    public void clearAll(boolean preserveSimple) {
        for (ServerCache cache : caches.values()) {
            if (cache instanceof SimpleCache) {
                if (BooleanUtils.isFalse(preserveSimple)) {
                    ((SimpleCache) cache).onClear();
                }
            } else if (cache instanceof LocalEbeanCache) {
                ((LocalEbeanCache) cache).onClear();
            }
        }
    }
}
