package com.turbospaces.ebean;

import java.math.BigDecimal;
import java.time.Duration;
import java.util.Collections;
import java.util.Set;

import javax.inject.Inject;
import javax.sql.DataSource;

import org.awaitility.Awaitility;
import org.awaitility.core.ThrowingRunnable;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Configurable;
import org.springframework.cloud.Cloud;
import org.springframework.cloud.service.ServiceInfo;
import org.springframework.context.annotation.Bean;

import com.turbospaces.boot.MockCloud;
import com.turbospaces.boot.SimpleBootstrap;
import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.ebean.query.QAccount;
import com.turbospaces.jdbc.HikariDataSourceFactoryBean;
import com.turbospaces.plugins.FlywayBootstrapInitializer;
import com.turbospaces.plugins.JGroupsBootstrapInitializer;
import com.turbospaces.ups.H2ServiceInfo;
import com.turbospaces.ups.UPSs;

import io.ebean.BackgroundExecutor;
import io.ebean.Database;
import io.ebean.DuplicateKeyException;
import io.ebean.Transaction;
import io.ebean.cache.ServerCache;
import io.ebean.cache.ServerCacheFactory;
import io.ebean.cache.ServerCachePlugin;
import io.ebean.cache.ServerCacheType;
import io.ebean.config.DatabaseConfig;
import io.ebean.config.dbplatform.h2.H2Platform;

public class EbeanInfinispanPluginTest {
    @Test
    public void works() throws Throwable {
        MockCloud builder = MockCloud.newMock();
        builder.withH2(true);
        ApplicationConfig cfg = builder.build();
        ApplicationProperties props = new ApplicationProperties(cfg);

        EbeanCacheConfigurer mngr = new DefaultEbeanCacheConfigurer(props);
        mngr.setMaxSizeLocal(Account.class, 1);
        mngr.setMaxSizeQuery(Account.class, 1);
        mngr.setMaxSizeLocal(Account.class, QAccount.alias().balances, 1);
        mngr.setMaxSizeReplicated(UTMTemplate.class, Byte.SIZE);
        mngr.setMaxSizeReplicated(AccountBalance.class, 2);
        mngr.setMaxSizeReplicated(AccountBalanceSnapshot.class, 4);
        mngr.setMaxSizeReplicated(Account.class, QAccount.alias().utmTemplates, Integer.MAX_VALUE);

        SimpleBootstrap bootstrap = new SimpleBootstrap(new ApplicationProperties(cfg), AppConfig.class);
        H2ServiceInfo ownerUps = UPSs.findRequiredServiceInfoByName(bootstrap, UPSs.H2_OWNER);

        bootstrap.addBootstrapRegistryInitializer(new JGroupsBootstrapInitializer(false));
        bootstrap.addBootstrapRegistryInitializer(new FlywayBootstrapInitializer(ownerUps, "CORE"));
        bootstrap.addBootstrapRegistryInitializer(new CacheBootstrapInitializer());
        bootstrap.run();

        try {

        } finally {
            bootstrap.shutdown();
        }
    }

    private static class RollbackBean extends AbstractEbeanAwareService implements InitializingBean {
        private final CacheManager manager;

        @Inject
        public RollbackBean(Database ebean, CacheManager manager) {
            super(ebean);
            this.manager = manager;
        }
        @Override
        public void afterPropertiesSet() throws Exception {
            String tname = UTMTemplate.class.getName();
            long id = Math.abs(hashCode());

            Account account;
            try (Transaction tx = newTransaction("cache-bean-save")) {
                account = new Account();

                account.setId(id);
                account.setFraud(new FraudJson(Collections.emptyMap()));
                account.setUsername("username_" + account.getId());
                account.setFirstName("f_" + account.getId());
                account.setLastName("l_" + account.getId());
                account.setDetails(Collections.emptyMap());

                ebean.save(account);
                tx.flush();

                UTMTemplate template1 = new UTMTemplate(account, "utm-1");
                ebean.save(template1, tx);

                UTMTemplate template2 = new UTMTemplate(account, "utm-2");
                ebean.save(template2, tx);

                ebean.createQuery(UTMTemplate.class).usingTransaction(tx).where().eq("account", account).eq("campaign", template1.getCampaign()).findOne();
                ebean.createQuery(UTMTemplate.class).usingTransaction(tx).where().eq("account", account).eq("campaign", template2.getCampaign()).findOne();

                UTMTemplate template3 = new UTMTemplate(account, "utm-2");
                ebean.save(template3, tx);

                tx.rollback();
            } catch (DuplicateKeyException err) {
                logger.warn(err.getMessage(), err);
            }

            ServerCache tCache = manager.cache(tname + ServerCacheType.BEAN.code());

            Assertions.assertEquals(2, tCache.size());
        }
    }

    public static class CacheBean extends AbstractEbeanAwareService implements InitializingBean {
        private final CacheManager manager;

        @Inject
        public CacheBean(Database ebean, CacheManager manager) {
            super(ebean);
            this.manager = manager;
        }
        @Override
        public void afterPropertiesSet() throws Exception {
            String aname = Account.class.getName();
            String abname = AccountBalance.class.getName();

            long id = Math.abs(hashCode());
            Account account;
            try (Transaction tx = newTransaction("cache-bean-save")) {
                account = new Account();

                account.setId(id);
                account.setFraud(new FraudJson(Collections.emptyMap()));
                account.setUsername("username_" + account.getId());
                account.setFirstName("f_" + account.getId());
                account.setLastName("l_" + account.getId());
                account.setDetails(Collections.emptyMap());

                AccountBalance balance1 = new AccountBalance(account, "USD");
                balance1.setAmount(BigDecimal.TEN);
                account.getBalances().add(balance1);

                AccountBalance balance2 = new AccountBalance(account, "EUR");
                balance2.setAmount(BigDecimal.ONE);
                account.getBalances().add(balance2);

                AccountBalance balance3 = new AccountBalance(account, "UAH");
                balance3.setAmount(BigDecimal.ONE);
                account.getBalances().add(balance3);

                ebean.save(account);
                tx.commit();
            }

            try (Transaction tx = newTransaction("cache-bean-save-all")) {
                for (Account it : ebean.find(Account.class).findList()) {
                    it.setAge(it.getAge() + 1);
                    it.setGameplayInfo(new GameplayInfo());
                    ebean.save(it, tx);
                }

                tx.commit();
            }

            for (int i = 0; i < 10; i++) {
                logger.debug("it ::: {}", i);
                try (Transaction tx = newReadOnlyTransaction("cache-bean-readonly")) {
                    tx.setReadOnly(true);
                    account = ebean.find(Account.class, id);
                    account.getBalances().size(); // id cache
                    for (AccountBalance balance : account.getBalances()) {
                        balance.getAmount().toString(); // balance cache
                    }
                }
            }

            for (int i = 0; i < 10; i++) {
                try (Transaction tx = newReadOnlyTransaction("cache-bean-readonly")) {
                    tx.setReadOnly(true);
                    Set<Account> accounts = ebean.createQuery(Account.class).setUseQueryCache(true).findSet();
                    accounts.size();
                }
            }
            // Assert.assertEquals( 1, qCache.size() );

            try (Transaction tx = newTransaction("cache-bean-save")) {
                account = ebean.find(Account.class, id);
                account.setAge(32);
                ebean.save(account);
                tx.commit();
            }
            Awaitility.await().atMost(Duration.ofSeconds(10)).untilAsserted(new ThrowingRunnable() {
                @Override
                public void run() throws Throwable {
                    // Assert.assertEquals( 0, qCache.size() );
                }
            });

            // Cache<Object, Object> qCache = channel.getCache( aname + ServerCacheType.QUERY.code() );
            ServerCache aCache = manager.cache(aname + ServerCacheType.BEAN.code());
            ServerCache bCache = manager.cache(abname + ServerCacheType.BEAN.code());
            ServerCache cCache = manager.cache(aname + "." + QAccount.alias().balances.toString() + ServerCacheType.COLLECTION_IDS.code());

            Assertions.assertEquals(1, aCache.size());
            Assertions.assertEquals(2, bCache.size());
            Assertions.assertEquals(1, cCache.size());

            ebean.cacheManager().beanCache(Account.class).clear();
            Assertions.assertEquals(0, aCache.size());
            Assertions.assertEquals(2, bCache.size()); // ~ ebean is not clearing children
            Assertions.assertEquals(1, cCache.size());

            // clear bean cache
            ebean.cacheManager().beanCache(AccountBalance.class).clear();
            Assertions.assertEquals(0, bCache.size());

            ebean.cacheManager().clear(Account.class);
            Assertions.assertEquals(0, cCache.size());
        }
    }

    public static class CacheCleanBean extends AbstractEbeanAwareService implements InitializingBean {
        private final CacheManager manager;

        @Inject
        public CacheCleanBean(Database ebean, CacheManager manager) {
            super(ebean);
            this.manager = manager;
        }
        @Override
        public void afterPropertiesSet() throws Exception {
            String aname = Account.class.getName();
            String abname = AccountBalance.class.getName();

            ServerCache aCache = manager.cache(aname + ServerCacheType.BEAN.code());
            ServerCache bCache = manager.cache(abname + ServerCacheType.BEAN.code());

            long id = Math.abs(hashCode());
            Account account;
            try (Transaction tx = newTransaction("save")) {
                account = new Account();

                account.setId(id);
                account.setUsername("username_" + account.getId());
                account.setFirstName("f_" + account.getId());
                account.setLastName("l_" + account.getId());

                AccountBalance balance1 = new AccountBalance(account, "USD");
                balance1.setAmount(BigDecimal.TEN);
                account.getBalances().add(balance1);

                AccountBalance balance2 = new AccountBalance(account, "EUR");
                balance2.setAmount(BigDecimal.ONE);
                account.getBalances().add(balance2);

                ebean.save(account);
                tx.commit();
            }

            for (int i = 0; i < 10; i++) {
                logger.debug("it ::: {}", i);
                try (Transaction tx = ebean.beginTransaction()) {
                    account = ebean.find(Account.class, id);
                    account.getBalances().size(); // id cache
                    for (AccountBalance balance : account.getBalances()) {
                        balance.getAmount().toString(); // balance cache
                    }
                }
            }

            CacheCleanTask action = new CacheCleanTask(ebean, Account.class, account.getId(), QAccount.alias().balances);
            action.run();
            Assertions.assertEquals(0, aCache.size());
            Assertions.assertEquals(2, bCache.size());
        }
    }

    @Configurable
    public static class AppConfig {
        @Bean
        public HikariDataSourceFactoryBean ds(Cloud cloud) {
            ServiceInfo appUps = UPSs.findRequiredServiceInfoByName(cloud, UPSs.H2_APP);
            return new HikariDataSourceFactoryBean(appUps);
        }
        @Bean
        public EbeanFactoryBean ebean(DataSource ds, CacheManager cache) {
            return new EbeanFactoryBean(ds) {
                @Override
                protected void configure(DatabaseConfig config) {
                    config.setDatabasePlatform(new H2Platform());
                    config.setServerCachePlugin(new ServerCachePlugin() {
                        @Override
                        public ServerCacheFactory create(DatabaseConfig dc, BackgroundExecutor executor) {
                            return cache;
                        }
                    });
                    config.addClass(Account.class);
                    config.addClass(GameplayInfo.class);
                    config.addClass(FraudJson.class);
                    config.addClass(AccountBalance.class);
                    config.addClass(AccountBalanceId.class);
                    config.addClass(AccountBalanceSnapshot.class);
                    config.addClass(AccountBalanceSnapshotId.class);
                    config.addClass(UTMTemplate.class);
                }
            };
        }
        @Bean
        public RollbackBean rollbackBean(EbeanFactoryBean ebean, CacheManager manager) throws Exception {
            return new RollbackBean(ebean.getObject(), manager);
        }
        @Bean
        public CacheBean cacheBean(EbeanFactoryBean ebean, CacheManager manager) throws Exception {
            return new CacheBean(ebean.getObject(), manager);
        }
        @Bean
        public CacheCleanBean cacheCleanBean(EbeanFactoryBean ebean, CacheManager manager) throws Exception {
            return new CacheCleanBean(ebean.getObject(), manager);
        }
    }
}
