package com.turbospaces.ebean;

import java.math.BigDecimal;
import java.time.LocalDate;

import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Configurable;
import org.springframework.cloud.Cloud;
import org.springframework.cloud.service.ServiceInfo;
import org.springframework.context.annotation.Bean;

import com.turbospaces.boot.MockCloud;
import com.turbospaces.boot.SimpleBootstrap;
import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.ebean.query.QAccount;
import com.turbospaces.ebean.query.QAccountBalance;
import com.turbospaces.jdbc.HikariDataSourceFactoryBean;
import com.turbospaces.plugins.FlywayBootstrapInitializer;
import com.turbospaces.plugins.JGroupsBootstrapInitializer;
import com.turbospaces.ups.UPSs;

import io.ebean.BackgroundExecutor;
import io.ebean.Database;
import io.ebean.cache.ServerCacheFactory;
import io.ebean.cache.ServerCachePlugin;
import io.ebean.config.DatabaseConfig;
import io.ebean.config.dbplatform.h2.H2Platform;

public class CompositeKeyTest {
    @Test
    public void works() throws Throwable {
        MockCloud builder = MockCloud.newMock();
        builder.withH2(true);
        ApplicationConfig cfg = builder.build();
        ApplicationProperties props = new ApplicationProperties(cfg);

        EbeanCacheConfigurer mngr = new DefaultEbeanCacheConfigurer(props);
        mngr.setMaxSizeLocal(Account.class, 1);
        mngr.setMaxSizeQuery(Account.class, 1);
        mngr.setMaxSizeLocal(Account.class, QAccount.alias().balances, 1);
        mngr.setMaxSizeReplicated(UTMTemplate.class, Byte.SIZE);
        mngr.setMaxSizeReplicated(AccountBalance.class, 10);
        mngr.setMaxSizeReplicated(AccountBalance.class, QAccountBalance.alias().snapshots, 10);
        mngr.setMaxSizeReplicated(AccountBalanceSnapshot.class, 30);
        mngr.setMaxSizeReplicated(Account.class, QAccount.alias().utmTemplates, Integer.MAX_VALUE);

        SimpleBootstrap bootstrap = new SimpleBootstrap(AppConfig.class, new ApplicationProperties(cfg));
        ServiceInfo ownerUps = UPSs.findRequiredServiceInfoByName(bootstrap, UPSs.H2_OWNER);

        bootstrap.addBootstrapRegistryInitializer(new FlywayBootstrapInitializer(ownerUps, "CORE"));
        bootstrap.addBootstrapRegistryInitializer(new JGroupsBootstrapInitializer(false));
        bootstrap.addBootstrapRegistryInitializer(new CacheBootstrapInitializer());
        bootstrap.run();

        Database ebean = bootstrap.context().getBean(Database.class);

        Account account = new Account();
        account.setId(System.currentTimeMillis());
        account.setUsername("username_" + account.getId());
        account.setFirstName("f_" + account.getId());
        account.setLastName("l_" + account.getId());
        account.setAge(18);
        account.setBirthDate(LocalDate.now().minusYears(account.getAge()));
        ebean.save(account);

        AccountBalance balance = new AccountBalance(account, "USD");
        balance.setAmount(BigDecimal.ONE);

        AccountBalanceSnapshot snapshot1 = new AccountBalanceSnapshot(balance, LocalDate.now());
        AccountBalanceSnapshot snapshot2 = new AccountBalanceSnapshot(balance, LocalDate.now().plusDays(1));

        balance.getSnapshots().add(snapshot1);
        balance.getSnapshots().add(snapshot2);
        ebean.save(balance);

        QAccountBalance q = new QAccountBalance(ebean);
        q.setId(balance.getPk());
        q.findList();

        try {

        } finally {
            bootstrap.shutdown();
        }
    }

    @Configurable
    public static class AppConfig {
        @Bean
        public HikariDataSourceFactoryBean ds(Cloud cloud) {
            ServiceInfo appUps = UPSs.findRequiredServiceInfoByName(cloud, UPSs.H2_APP);
            return new HikariDataSourceFactoryBean(appUps);
        }
        @Bean
        public EbeanFactoryBean ebean(HikariDataSourceFactoryBean ds, CacheManager cache) throws Exception {
            return new EbeanFactoryBean(ds.getObject()) {
                @Override
                protected void configure(DatabaseConfig config) {
                    config.setDatabasePlatform(new H2Platform());
                    config.setServerCachePlugin(new ServerCachePlugin() {
                        @Override
                        public ServerCacheFactory create(DatabaseConfig dc, BackgroundExecutor executor) {
                            return cache;
                        }
                    });
                    config.addClass(Account.class);
                    config.addClass(GameplayInfo.class);
                    config.addClass(FraudJson.class);
                    config.addClass(UTMTemplate.class);
                    config.addClass(AccountBalance.class);
                    config.addClass(AccountBalanceId.class);
                    config.addClass(AccountBalanceSnapshot.class);
                    config.addClass(AccountBalanceSnapshotId.class);
                }
            };
        }
    }
}
