package com.turbospaces.ebean;

import org.jgroups.Address;
import org.jgroups.JChannel;
import org.jgroups.MembershipListener;
import org.jgroups.View;
import org.jgroups.blocks.RequestOptions;
import org.jgroups.blocks.RpcDispatcher;
import org.springframework.beans.factory.support.DefaultListableBeanFactory;
import org.springframework.beans.factory.support.GenericBeanDefinition;
import org.springframework.boot.BootstrapContextClosedEvent;
import org.springframework.boot.BootstrapRegistry;
import org.springframework.boot.BootstrapRegistry.InstanceSupplier;
import org.springframework.boot.BootstrapRegistryInitializer;
import org.springframework.boot.context.event.ApplicationFailedEvent;
import org.springframework.context.ApplicationEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.event.SmartApplicationListener;

import com.turbospaces.boot.AbstractBootstrapAware;
import com.turbospaces.boot.Bootstrap;

public class CacheBootstrapInitializer extends AbstractBootstrapAware implements BootstrapRegistryInitializer {
    private final RpcDispatcher dispatcher = new RpcDispatcher();
    private DefaultEbeanCacheManager manager;

    @Override
    public void setBootstrap(Bootstrap bootstrap) {
        super.setBootstrap( bootstrap );

        manager = new DefaultEbeanCacheManager( dispatcher, bootstrap.isDevMode() ? RequestOptions.SYNC() : RequestOptions.ASYNC() );
        manager.setBootstrap( bootstrap );
    }
    @Override
    public void initialize(BootstrapRegistry registry) {
        registry.register( CacheManager.class, InstanceSupplier.of( manager ) );
        registry.addCloseListener( new ApplicationListener<BootstrapContextClosedEvent>() {
            @Override
            public void onApplicationEvent(BootstrapContextClosedEvent event) {
                InstanceSupplier<JChannel> supplier = registry.getRegisteredInstanceSupplier( JChannel.class );
                JChannel jchannel = supplier.get( event.getBootstrapContext() );

                ConfigurableApplicationContext applicationContext = event.getApplicationContext();
                applicationContext.addApplicationListener( new SmartApplicationListener() {
                    @Override
                    public void onApplicationEvent(ApplicationEvent e) {
                        dispatcher.stop();
                    }
                    @Override
                    public boolean supportsEventType(Class<? extends ApplicationEvent> eventType) {
                        return eventType.equals( ApplicationFailedEvent.class );
                    }
                } );

                dispatcher.setChannel( jchannel );
                dispatcher.setServerObject( manager );

                dispatcher.setMethodLookup( manager );
                dispatcher.setMembershipListener( new MembershipListener() {
                    @Override
                    public void viewAccepted(View view) {
                        logger.debug( "new view: {}", view );
                    }
                    @Override
                    public void suspect(Address member) {
                        logger.debug( "suspected member: {}", member );
                    }
                } );
                dispatcher.start();

                DefaultListableBeanFactory beanFactory = (DefaultListableBeanFactory) applicationContext.getBeanFactory();
                beanFactory.registerSingleton( "rpc-dispatcher", dispatcher );
                beanFactory.registerSingleton( "cache-manager", manager );

                // ~ shutdown gracefully
                GenericBeanDefinition bean = new GenericBeanDefinition();
                bean.setBeanClass( RpcDispatcherDisposableBean.class );
                bean.setAutowireCandidate( true );
                beanFactory.registerBeanDefinition( "rpc-dispatcher-lifecycle", bean );
            }
        } );
    }
}
