package com.turbospaces.ebean;

import java.util.Objects;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheStats;

import io.ebean.cache.ServerCacheStatistics;

public class SimpleCache implements LocalCache, Supplier<Cache<Object, Object>> {
    private final Logger logger = LoggerFactory.getLogger( getClass() );
    private final String cacheKey;
    private final Cache<Object, Object> cache;

    public SimpleCache(String cacheKey) {
        this.cacheKey = Objects.requireNonNull( cacheKey );
        this.cache = CacheBuilder.newBuilder().recordStats().build();
    }
    @Override
    public Object get(Object id) {
        return cache.getIfPresent( id );
    }
    @Override
    public void put(Object id, Object value) {
        cache.put( id, value );
    }
    @Override
    public void remove(Object id) {
        cache.invalidate( id );
    }
    @Override
    public int size() {
        return (int) cache.size();
    }
    @Override
    public int getHitRatio() {
        return (int) ( cache.stats().hitRate() * 100 );
    }
    @Override
    public ServerCacheStatistics getStatistics(boolean reset) {
        CacheStats stats = cache.stats();

        ServerCacheStatistics scs = new ServerCacheStatistics();
        scs.setCacheName( cacheKey );
        scs.setEvictCount( stats.evictionCount() );
        scs.setHitCount( stats.hitCount() );
        scs.setMissCount( stats.missCount() );
        scs.setSize( size() );

        return scs;
    }
    @Override
    public void clear() {
        logger.debug( "clearing {}(items={})", cacheKey, cache.size() );
        cache.invalidateAll();
        cache.cleanUp();
    }
    @Override
    public int onClear() {
        long size = cache.size();
        clear();
        return (int) size;
    }
    @Override
    public Cache<Object, Object> get() {
        return cache;
    }
    @Override
    public String toString() {
        return cacheKey;
    }
}
