package com.turbospaces.ebean;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.builder.ReflectionToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.jgroups.blocks.MethodCall;
import org.jgroups.blocks.RequestOptions;
import org.jgroups.blocks.RpcDispatcher;
import org.jgroups.util.RspList;

import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.netflix.archaius.api.Config;
import com.turbospaces.boot.AbstractBootstrapAware;

import io.ebean.cache.QueryCacheEntryValidate;
import io.ebean.cache.ServerCache;
import io.ebean.cache.ServerCacheConfig;
import io.ebean.cache.ServerCacheNotification;
import io.ebean.cache.ServerCacheNotify;
import io.ebean.cache.ServerCacheOptions;
import io.ebean.cache.ServerCacheType;
import io.ebean.config.CurrentTenantProvider;
import io.ebeaninternal.server.cache.DefaultServerCacheConfig;
import io.ebeaninternal.server.cache.DefaultServerQueryCache;

public class DefaultEbeanCacheManager extends AbstractBootstrapAware implements ServerCacheNotify, CacheManager {
    public static ImmutableMap<Short, Method> METHODS;

    static {
        try {
            ImmutableMap.Builder<Short, Method> b = ImmutableMap.builder();
            b.put( METHOD_ON_MODIFIED, DefaultEbeanCacheManager.class.getMethod( "onTablesModify", String.class ) );
            b.put( METHOD_ON_CACHE_PUT, DefaultEbeanCacheManager.class.getMethod( "onCachePut", String.class, byte[].class, byte[].class ) );
            b.put( METHOD_ON_CHANGE_REMOVE, DefaultEbeanCacheManager.class.getMethod( "onCacheRemove", String.class, byte[].class ) );
            b.put( METHOD_ON_CACHE_CLEAR, DefaultEbeanCacheManager.class.getMethod( "onCacheClear", String.class ) );
            b.put( METHOD_ON_CACHE_CLEAR_ALL, DefaultEbeanCacheManager.class.getMethod( "onCacheClearAll", boolean.class ) );

            METHODS = b.build();
        }
        catch ( NoSuchMethodException err ) {
            throw new Error( err );
        }
    }

    private final Map<String, ServerCache> caches = new ConcurrentHashMap<>();
    private final RpcDispatcher dispatcher;
    private final RequestOptions requestOptions;
    private ServerCacheNotify notify;

    public DefaultEbeanCacheManager(RpcDispatcher dispatcher, RequestOptions requestOptions) {
        this.dispatcher = Objects.requireNonNull( dispatcher );
        this.requestOptions = Objects.requireNonNull( requestOptions );
    }
    @Override
    public Method findMethod(short id) {
        return METHODS.get( id );
    }
    @Override
    public ServerCache cache(String cacheKey) {
        return caches.get( cacheKey );
    }
    @Override
    public ServerCache createCache(ServerCacheConfig config) {
        return getCache( config );
    }
    @Override
    public ServerCacheNotify createCacheNotify(ServerCacheNotify cacheNotify) {
        this.notify = Objects.requireNonNull( cacheNotify );
        return this;
    }
    @Override
    public void notify(ServerCacheNotification notification) {
        if ( notification.getDependentTables() != null && !notification.getDependentTables().isEmpty() ) {
            Set<String> dependentTables = notification.getDependentTables();
            String line = Joiner.on( ',' ).join( dependentTables );
            logger.debug( "Publish TableMods - {}", line );

            try {
                MethodCall call = new MethodCall( METHOD_ON_MODIFIED, line );
                RspList<Object> l = dispatcher.callRemoteMethods( null, call, requestOptions );
                if ( l != null ) {
                    CacheManager.trace( logger, l );
                }
            }
            catch ( Throwable t ) {
                throw new RuntimeException( t );
            }
        }
    }
    @Override
    public void onTablesModify(String line) {
        Iterable<String> it = Splitter.on( ',' ).omitEmptyStrings().split( line );
        ImmutableSet<String> tables = ImmutableSet.copyOf( it );
        ServerCacheNotification notification = new ServerCacheNotification( tables );
        notify.notify( notification );
    }
    @Override
    public void onCachePut(String cacheKey, byte[] id, byte[] value) {
        ReplicatedEbeanCache cache = (ReplicatedEbeanCache) caches.get( cacheKey );
        if ( cache != null ) {
            cache.onPut( id, value );
        }
    }
    @Override
    public void onCacheRemove(String cacheKey, byte[] id) {
        ReplicatedEbeanCache cache = (ReplicatedEbeanCache) caches.get( cacheKey );
        if ( cache != null ) {
            cache.onRemove( id );
        }
    }
    @Override
    public void onCacheClear(String cacheKey) {
        ServerCache cache = caches.get( cacheKey );
        if ( cache != null ) {
            if ( cache instanceof ReplicatedEbeanCache ) {
                ( (ReplicatedEbeanCache) cache ).onClear(); // ~ only for replicated scenarios
            }
        }
    }
    @Override
    public void onCacheClearAll(boolean preserveSimple) {
        synchronized ( this ) {
            for ( ServerCache cache : caches.values() ) {
                if ( cache instanceof ReplicatedEbeanCache ) {
                    ( (ReplicatedEbeanCache) cache ).onClear();
                }
                else if ( cache instanceof SimpleCache ) {
                    if ( preserveSimple ) {}
                    else {
                        ( (SimpleCache) cache ).onClear();
                    }
                }
                else if ( cache instanceof LocalEbeanCache ) {
                    ( (LocalEbeanCache) cache ).onClear();
                }
            }
        }
    }
    @Override
    public SimpleCache simpleCache(String name) {
        synchronized ( this ) {
            ServerCache cache = caches.get( name );
            if ( cache == null ) {
                cache = new SimpleCache( name );
                caches.put( name, cache );
            }
            return (SimpleCache) cache;
        }
    }
    @Override
    public void clearAllLocal() {
        synchronized ( this ) {
            for ( ServerCache cache : caches.values() ) {
                if ( cache instanceof ReplicatedEbeanCache ) {} // ~ ignore replicated
                else if ( cache instanceof SimpleCache ) {} // ~ ignore simple
                else if ( cache instanceof LocalEbeanCache ) {
                    ( (LocalEbeanCache) cache ).onClear();
                }
            }
        }
    }
    @Override
    public void clearAllSimple() {
        synchronized ( this ) {
            for ( ServerCache cache : caches.values() ) {
                if ( cache instanceof ReplicatedEbeanCache ) {} // ~ ignore replicated
                else if ( cache instanceof LocalEbeanCache ) {} // ~ ignore local
                else if ( cache instanceof SimpleCache ) {
                    ( (SimpleCache) cache ).onClear();
                }
            }
        }
    }
    @Override
    public void clearAll(boolean preserveSimple) {
        try {
            MethodCall call = new MethodCall( CacheManager.METHOD_ON_CACHE_CLEAR_ALL, preserveSimple );
            RspList<Object> l = dispatcher.callRemoteMethods( null, call, requestOptions );
            if ( l != null ) {
                CacheManager.trace( logger, l );
            }
        }
        catch ( Throwable t ) {
            throw new RuntimeException( t );
        }
        finally {
            onCacheClearAll( preserveSimple );
        }
    }
    private ServerCache getCache(ServerCacheConfig config) {
        String cacheKey = config.getCacheKey();
        String shortName = config.getShortName();

        ServerCacheOptions cacheOptions = config.getCacheOptions();
        ServerCacheType cacheType = config.getType();
        CurrentTenantProvider tenantProvider = config.getTenantProvider();
        QueryCacheEntryValidate queryCacheValidate = config.getQueryCacheEntryValidate();

        synchronized ( this ) {
            ServerCache cache = caches.get( cacheKey );
            if ( cache == null ) {
                Config prefixedView = bootstrap.cfg().getPrefixedView( cacheKey );
                Iterator<String> it = prefixedView.getKeys();
                Map<String, Object> configMap = new HashMap<>();
                while ( it.hasNext() ) {
                    String key = it.next();
                    Object rawProperty = prefixedView.getRawProperty( key );
                    configMap.put( key, rawProperty );
                }

                if ( prefixedView.isEmpty() ) {
                    throw new IllegalArgumentException( "no cache defined for key: " + cacheKey );
                }

                int maxTtl = prefixedView.getInteger( EbeanCacheConfigurer.MAX_TTL, cacheOptions.getMaxSecsToLive() );
                int maxIdle = prefixedView.getInteger( EbeanCacheConfigurer.MAX_IDLE, cacheOptions.getMaxIdleSecs() );
                int maxSize = prefixedView.getInteger( EbeanCacheConfigurer.MAX_SIZE, cacheOptions.getMaxSize() );
                int trimFrequency = bootstrap.props().APP_TIMER_INTERVAL.get();

                ServerCacheOptions options = new ServerCacheOptions();
                options.setMaxSecsToLive( maxTtl );
                options.setMaxIdleSecs( maxIdle );
                options.setMaxSize( maxSize );
                options.setTrimFrequency( trimFrequency );

                ServerCacheConfig scc = new ServerCacheConfig( cacheType, cacheKey, shortName, options, tenantProvider, queryCacheValidate );

                if ( config.isQueryCache() ) {
                    logger.debug( "creating query cache: {} using cfg {}", cacheKey, ReflectionToStringBuilder.toString( options, ToStringStyle.NO_CLASS_NAME_STYLE ) );
                    cache = new DefaultServerQueryCache( new DefaultServerCacheConfig( scc ) );
                }
                else {
                    boolean localMode = prefixedView.getBoolean( EbeanCacheConfigurer.CACHE_MODE_LOCAL, true );

                    if ( localMode ) {
                        boolean neverExpire = bootstrap.props().CACHE_LOCAL_NEVER_EXPIRE.get();

                        if ( neverExpire ) {
                            logger.debug( "creating local cache: {} with max size: {}", cacheKey, maxSize );
                            cache = new LocalEbeanCache( cacheKey, maxSize, tenantProvider );
                        }
                        else {
                            logger.debug( "creating local cache: {} using cfg {}", cacheKey, ReflectionToStringBuilder.toString( options, ToStringStyle.NO_CLASS_NAME_STYLE ) );
                            cache = new LocalEbeanCache( cacheKey, new DefaultServerCacheConfig( scc ), tenantProvider );
                        }
                    }
                    else {
                        LocalCache local;
                        boolean neverExpire = bootstrap.props().CACHE_REPLICATED_NEVER_EXPIRE.get();

                        if ( neverExpire ) {
                            logger.debug( "creating replicated cache: {} with max size: {}", cacheKey, maxSize );
                            local = new LocalEbeanCache( cacheKey, maxSize, tenantProvider );
                        }
                        else {
                            logger.debug( "creating replicated cache: {} using cfg {}", cacheKey, ReflectionToStringBuilder.toString( options, ToStringStyle.NO_CLASS_NAME_STYLE ) );
                            local = new LocalEbeanCache( cacheKey, new DefaultServerCacheConfig( scc ), tenantProvider );
                        }

                        cache = new ReplicatedEbeanCache( cacheKey, dispatcher, requestOptions, local, scc );
                    }
                }

                caches.put( cacheKey, cache );

                // ~ cleanUp
                ServerCache tmp = cache;
                bootstrap.platform().scheduleWithFixedDelay( new Runnable() {
                    @Override
                    public void run() {
                        if ( tmp instanceof LocalEbeanCache ) {
                            ( (LocalEbeanCache) tmp ).get().cleanUp();
                        }
                        else if ( tmp instanceof DefaultServerQueryCache ) {
                            ( (DefaultServerQueryCache) tmp ).runEviction();
                        }
                    }
                }, trimFrequency, trimFrequency, TimeUnit.SECONDS );
            }
            return cache;
        }
    }
}
