package com.turbospaces.ebean;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.Objects;

import org.apache.commons.lang3.SerializationUtils;
import org.jgroups.blocks.MethodCall;
import org.jgroups.blocks.RequestOptions;
import org.jgroups.blocks.RpcDispatcher;
import org.jgroups.util.RspList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.ebean.cache.ServerCache;
import io.ebean.cache.ServerCacheConfig;
import io.ebean.cache.ServerCacheStatistics;
import io.ebean.cache.ServerCacheType;
import io.ebeaninternal.server.cache.CachedBeanData;
import io.ebeaninternal.server.cache.CachedManyIds;

public class ReplicatedEbeanCache implements ReplicatedCache {
    private final Logger logger = LoggerFactory.getLogger(getClass());
    private final String cacheKey;
    private final RpcDispatcher dispatcher;
    private final RequestOptions requestOptions;
    private final ServerCache local;
    private final ServerCacheConfig config;

    public ReplicatedEbeanCache(String cacheKey, RpcDispatcher dispatcher, RequestOptions requestOptions, ServerCache local, ServerCacheConfig config) {
        this.cacheKey = Objects.requireNonNull(cacheKey);
        this.dispatcher = Objects.requireNonNull(dispatcher);
        this.requestOptions = Objects.requireNonNull(requestOptions);
        this.local = Objects.requireNonNull(local);
        this.config = Objects.requireNonNull(config);
    }
    @Override
    public int size() {
        return local.size();
    }
    @Override
    public Object get(Object id) {
        return local.get(id);
    }
    @Override
    public int hitRatio() {
        return local.hitRatio();
    }
    @Override
    public ServerCacheStatistics statistics(boolean reset) {
        return local.statistics(reset);
    }
    @Override
    public void put(Object id, Object value) {
        local.put(id, value);

        byte[] keyAsBytes = SerializationUtils.serialize((Serializable) id);
        ByteArrayOutputStream out = new ByteArrayOutputStream();

        switch (config.getType()) {
            case NATURAL_KEY: {
                try (ObjectOutputStream oos = new ObjectOutputStream(out)) {
                    oos.writeObject(value);
                    oos.flush();
                } catch (IOException err) {
                    throw new RuntimeException("Failed to encode cache data", err);
                }

                break;
            }
            case BEAN: {
                try (ObjectOutputStream oos = new ObjectOutputStream(out)) {
                    CachedBeanData data = (CachedBeanData) value;
                    data.writeExternal(oos);
                    oos.flush();
                } catch (IOException err) {
                    throw new RuntimeException("Failed to encode bean cache data", err);
                }

                break;
            }
            case COLLECTION_IDS: {
                try (ObjectOutputStream oos = new ObjectOutputStream(out)) {
                    CachedManyIds data = (CachedManyIds) value;
                    data.writeExternal(oos);
                    oos.flush();
                } catch (IOException err) {
                    throw new RuntimeException("Failed to encode ids cache data", err);
                }

                break;
            }
            default: {
                throw new IllegalArgumentException("Unexpected cache type: " + config.getType());
            }
        }

        try {
            logger.debug("putting {} entry on remote nodes by key: {} value: {}", cacheKey, id, value);

            MethodCall call = new MethodCall(JGroupsCacheManager.METHOD_ON_CACHE_PUT, cacheKey, keyAsBytes, out.toByteArray());
            RspList<Object> l = dispatcher.callRemoteMethods(null, call, requestOptions);
            if (Objects.nonNull(l)) {
                ReplicatedCache.trace(logger, l);
            }
        } catch (Throwable t) {
            throw new RuntimeException(t);
        }
    }
    @Override
    public void remove(Object id) {
        local.remove(id);

        try {
            logger.debug("removing {} entry on remote nodes by key: {} ...", cacheKey, id);

            byte[] keyAsBytes = SerializationUtils.serialize((Serializable) id);
            MethodCall call = new MethodCall(JGroupsCacheManager.METHOD_ON_CHANGE_REMOVE, cacheKey, keyAsBytes);
            RspList<Object> l = dispatcher.callRemoteMethods(null, call, requestOptions);
            if (Objects.nonNull(l)) {
                ReplicatedCache.trace(logger, l);
            }
        } catch (Throwable t) {
            throw new RuntimeException(t);
        }
    }
    @Override
    public void clear() {
        local.clear();

        logger.debug("clearing {} on remote nodes ...", cacheKey);

        try {
            MethodCall call = new MethodCall(JGroupsCacheManager.METHOD_ON_CACHE_CLEAR, cacheKey);
            RspList<Object> l = dispatcher.callRemoteMethods(null, call, requestOptions);
            if (Objects.nonNull(l)) {
                ReplicatedCache.trace(logger, l);
            }
        } catch (Throwable t) {
            throw new RuntimeException(t);
        }
    }
    @Override
    public void onPut(byte[] keyData, byte[] valueData) {
        Object key = SerializationUtils.deserialize(keyData);
        ByteArrayInputStream is = new ByteArrayInputStream(valueData);

        switch (config.getType()) {
            case NATURAL_KEY: {
                try (ObjectInputStream ois = new ObjectInputStream(is)) {
                    Object read = ois.readObject();

                    logger.debug("onPut {} by key: {} value: {}", cacheKey, key, read);
                    local.put(key, read);
                } catch (ClassNotFoundException | IOException err) {
                    throw new RuntimeException("failed to decode cache data", err);
                }

                break;
            }
            case BEAN: {
                try (ObjectInputStream ois = new ObjectInputStream(is)) {
                    CachedBeanData read = new CachedBeanData();
                    read.readExternal(ois);

                    logger.debug("onPut {} by key: {} bean value: {}", cacheKey, key, read);
                    local.put(key, read);
                } catch (ClassNotFoundException | IOException err) {
                    throw new RuntimeException("failed to decode bean cache data", err);
                }

                break;
            }
            case COLLECTION_IDS: {
                try (ObjectInputStream ois = new ObjectInputStream(is)) {
                    CachedManyIds read = new CachedManyIds();
                    read.readExternal(ois);

                    logger.debug("onPut {} by key: {} ids value: {}", cacheKey, key, read);
                    local.put(key, read);
                } catch (ClassNotFoundException | IOException err) {
                    throw new RuntimeException("failed to decode ids cache data", err);
                }

                break;
            }
            default: {
                throw new IllegalArgumentException("unexpected cache type: " + config.getType());
            }
        }
    }
    @Override
    public void onRemove(byte[] data) {
        Object key = SerializationUtils.deserialize(data);

        logger.debug("onRemove {} by key: {}", cacheKey, key);
        local.remove(key);
    }
    @Override
    public int onClear() {
        logger.debug("onClear {}", cacheKey);
        int size = local.size();
        local.clear();
        return size;
    }
    @Override
    public ServerCacheType type() {
        return config.getType();
    }
}
