package com.turbospaces.debezium;

import java.io.IOException;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import javax.persistence.Table;

import org.apache.commons.lang3.BooleanUtils;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.connect.data.Struct;
import org.apache.kafka.connect.runtime.WorkerConfig;
import org.apache.kafka.connect.runtime.distributed.DistributedConfig;
import org.apache.kafka.connect.source.SourceRecord;
import org.apache.kafka.connect.storage.KafkaOffsetBackingStore;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.service.common.PostgresqlServiceInfo;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.SmartLifecycle;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.kafka.core.MicrometerProducerListener;
import org.springframework.kafka.support.SendResult;
import org.springframework.kafka.test.EmbeddedKafkaBroker;
import org.springframework.util.concurrent.ListenableFutureCallback;
import org.testcontainers.containers.PostgreSQLContainer;
import org.testcontainers.shaded.org.awaitility.Awaitility;

import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.turbospaces.boot.AbstractBootstrapAware;
import com.turbospaces.boot.Bootstrap;
import com.turbospaces.boot.MockCloud;
import com.turbospaces.boot.SimpleBootstrap;
import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.ebean.EbeanDatabaseConfig;
import com.turbospaces.ebean.EbeanJpaManager;
import com.turbospaces.ebean.FlywayUberRunner;
import com.turbospaces.ebean.JpaManager;
import com.turbospaces.jdbc.HikariDataSourceFactoryBean;
import com.turbospaces.kafka.KafkaProducerProperties;
import com.turbospaces.kafka.KafkaWithMetricsProducerFactory;
import com.turbospaces.ups.KafkaServiceInfo;
import com.turbospaces.ups.UPSs;

import io.debezium.connector.AbstractSourceInfo;
import io.debezium.connector.postgresql.PostgresConnector;
import io.debezium.connector.postgresql.PostgresConnectorConfig;
import io.debezium.connector.postgresql.PostgresConnectorConfig.AutoCreateMode;
import io.debezium.connector.postgresql.PostgresConnectorConfig.LogicalDecoder;
import io.debezium.connector.postgresql.PostgresConnectorConfig.SnapshotMode;
import io.debezium.data.Envelope;
import io.debezium.data.Envelope.Operation;
import io.debezium.embedded.Connect;
import io.debezium.embedded.EmbeddedEngine;
import io.debezium.engine.ChangeEvent;
import io.debezium.engine.DebeziumEngine;
import io.debezium.engine.DebeziumEngine.ChangeConsumer;
import io.debezium.engine.DebeziumEngine.RecordCommitter;
import io.debezium.relational.HistorizedRelationalDatabaseConnectorConfig;
import io.debezium.relational.RelationalDatabaseConnectorConfig;
import io.debezium.relational.history.DatabaseHistory;
import io.debezium.relational.history.KafkaDatabaseHistory;
import io.ebean.platform.postgres.Postgres9Platform;

public class DebeziumTest {
    public static Logger LOGGER = LoggerFactory.getLogger(DebeziumTest.class);

    public static final String TBL_ACCOUNT = Account.class.getAnnotation(Table.class).name();
    public static final String TBL_ACCOUNT_BALANCE = AccountBalance.class.getAnnotation(Table.class).name();
    public static final String SCHEMA_CORE = "core";

    @Test
    public void works() throws Exception {
        EmbeddedKafkaBroker broker = new EmbeddedKafkaBroker(1, false, Runtime.getRuntime().availableProcessors());
        broker.afterPropertiesSet();

        try (PostgreSQLContainer<?> pg = new PostgreSQLContainer<>("postgres:14")) {
            pg.setCommand("postgres", "-c", "wal_level=logical");
            pg.start();

            try {
                int brokerPort = Iterables.getOnlyElement(Arrays.asList(broker.getBrokerAddresses())).getPort();
                ApplicationConfig cfg = MockCloud.newMock().build();

                Bootstrap bootstrap = new SimpleBootstrap(new ApplicationProperties(cfg), AppConfig.class);
                bootstrap.cfg().setLocalProperty("kafka.port", brokerPort);
                bootstrap.cfg().setLocalProperty("db.host", pg.getHost());
                bootstrap.cfg().setLocalProperty("db.port", pg.getFirstMappedPort());
                bootstrap.cfg().setLocalProperty("db.name", pg.getDatabaseName());
                bootstrap.withKafka(brokerPort);
                ConfigurableApplicationContext applicationContext = bootstrap.run();

                JpaManager jpaManager = applicationContext.getBean(JpaManager.class);
                jpaManager.execute(new Runnable() {
                    @Override
                    public void run() {
                        for (int i = 1; i <= Byte.MAX_VALUE; i++) {
                            Account acc = new Account();
                            acc.setId(i);
                            acc.setUsername(UUID.randomUUID().toString());
                            acc.setBirthDate(LocalDate.now());
                            jpaManager.save(acc);

                            AccountBalance balance1 = new AccountBalance(acc, "USD");
                            balance1.setAmount(BigDecimal.ONE);
                            jpaManager.save(balance1);

                            AccountBalance balance2 = new AccountBalance(acc, "EUR");
                            balance2.setAmount(BigDecimal.TEN);
                            jpaManager.save(balance2);
                        }
                    }
                });

                // ~ start
                for (SmartLifecycle channel : applicationContext.getBeansOfType(SmartLifecycle.class).values()) {
                    if (BooleanUtils.isFalse(channel.isRunning())) {
                        channel.start();
                    }
                }

                // ~ await for publication to arrive
                Awaitility.await().until(new Callable<Boolean>() {
                    @Override
                    public Boolean call() throws Exception {
                        Long own = jpaManager.sqlQuery("select pubowner from pg_catalog.pg_publication").mapToScalar(Long.class).findOne();
                        if (Objects.nonNull(own)) {
                            LOGGER.info("pubowner: {}", own);
                        }
                        return Objects.nonNull(own);
                    }
                });

                BatchConsumerSingleRecord consumer1 = applicationContext.getBean(BatchConsumerSingleRecord.class);
                consumer1.genData1();
                Assertions.assertTrue(consumer1.await());

                bootstrap.shutdown();
            } finally {
                pg.stop();
            }
        } finally {
            broker.destroy();
        }
    }

    public static class BatchConsumerSingleRecord implements ChangeConsumer<ChangeEvent<SourceRecord, SourceRecord>> {
        private final CountDownLatch latch = new CountDownLatch(4); // account + 3 balances
        private final Long ID = System.currentTimeMillis();
        private final JpaManager jpaManager;
        private final KafkaTemplate<byte[], byte[]> kafkaTemplate;

        @Autowired
        public BatchConsumerSingleRecord(EbeanFactoryBean factory, KafkaTemplate<byte[], byte[]> kafkaTemplate) throws Exception {
            this.kafkaTemplate = kafkaTemplate;
            this.jpaManager = factory.getObject();
        }
        public void genData1() throws Exception {
            Account acc = new Account();
            acc.setId(ID);
            acc.setUsername(UUID.randomUUID().toString());
            acc.setBirthDate(LocalDate.now());
            jpaManager.save(acc);

            jpaManager.execute(new Runnable() {
                @Override
                public void run() {
                    AccountBalance balance1 = new AccountBalance(acc, "USD");
                    balance1.setAmount(BigDecimal.ONE);
                    jpaManager.save(balance1);

                    AccountBalance balance2 = new AccountBalance(acc, "EUR");
                    balance2.setAmount(BigDecimal.ONE);
                    jpaManager.save(balance2);

                    AccountBalance balance3 = new AccountBalance(acc, "UAH");
                    balance3.setAmount(BigDecimal.ONE);
                    jpaManager.save(balance3);
                }
            });
        }
        public boolean await() throws InterruptedException {
            return latch.await(30, TimeUnit.SECONDS);
        }
        @Override
        public void handleBatch(
                List<ChangeEvent<SourceRecord, SourceRecord>> records,
                RecordCommitter<ChangeEvent<SourceRecord, SourceRecord>> committer) throws InterruptedException {
            CountDownLatch n = new CountDownLatch(records.size());

            for (ChangeEvent<SourceRecord, SourceRecord> event : records) {
                SourceRecord value = event.value();
                Struct envelope = (Struct) value.value();

                LOGGER.trace("publishing change: {} to: {}", envelope.toString(), event.destination());
                kafkaTemplate.send(new ProducerRecord<byte[], byte[]>(event.destination(), null, envelope.toString().getBytes()))
                        .addCallback(new ListenableFutureCallback<SendResult<byte[], byte[]>>() {
                            @Override
                            public void onSuccess(SendResult<byte[], byte[]> result) {
                                try {
                                    if (Objects.nonNull(envelope)) {
                                        Operation operation = Operation.forCode((String) envelope.get(Envelope.FieldName.OPERATION));

                                        if (ImmutableSet.of(Operation.CREATE, Operation.READ).contains(operation)) {
                                            Struct source = (Struct) envelope.get(Envelope.FieldName.SOURCE);
                                            String schema = source.get(AbstractSourceInfo.SCHEMA_NAME_KEY).toString();
                                            String table = source.get(AbstractSourceInfo.TABLE_NAME_KEY).toString();

                                            if (Objects.nonNull(source)) {
                                                Struct after = (Struct) envelope.get(Envelope.FieldName.AFTER);
                                                if (Objects.nonNull(after)) {
                                                    if (SCHEMA_CORE.equals(schema) && TBL_ACCOUNT.equals(table)) {
                                                        Long id = after.getInt64("id");

                                                        LOGGER.debug("schema: {}, table: {}, id: {}", schema, table, id);

                                                        if (ID.equals(id)) {
                                                            latch.countDown();
                                                        }
                                                    } else if (SCHEMA_CORE.equals(schema) && TBL_ACCOUNT_BALANCE.equals(table)) {
                                                        Long accountId = after.getInt64("account_id");
                                                        String currency = after.getString("currency");
                                                        BigDecimal amount = (BigDecimal) after.get("amount");

                                                        LOGGER.debug("schema: {}, table: {}, account_id: {}, currency: {}, amount: {}",
                                                                schema,
                                                                table,
                                                                accountId,
                                                                currency,
                                                                amount);

                                                        if (ID.equals(after.getInt64("account_id"))) {
                                                            latch.countDown();
                                                        }
                                                    }
                                                }
                                            }
                                        }
                                    }
                                } finally {
                                    n.countDown();
                                }
                            }
                            @Override
                            public void onFailure(Throwable ex) {
                                LOGGER.error(ex.getMessage(), ex);
                            }
                        });
                committer.markProcessed(event);
            }

            n.await();
            committer.markBatchFinished();
        }
    }

    @Configuration
    public static class AppConfig extends AbstractBootstrapAware {
        public PostgresqlServiceInfo psi() {
            String dbName = bootstrap.cfg().getString("db.name");
            String dbHost = bootstrap.cfg().getString("db.host");
            int dbPort = bootstrap.cfg().getInteger("db.port");

            String jdbcUrl = String.format("postgres://test:test@%s:%d/%s", dbHost, dbPort, dbName);
            return new PostgresqlServiceInfo(UPSs.POSTGRES_OWNER, jdbcUrl);
        }
        @Bean
        public KafkaTemplate<byte[], byte[]> kafkaTemplate() {
            KafkaServiceInfo si = UPSs.findRequiredServiceInfoByName(bootstrap, UPSs.KAFKA);
            KafkaProducerProperties producerProps = new KafkaProducerProperties(si);
            producerProps.setBootstrap(bootstrap);

            MicrometerProducerListener<byte[], byte[]> metricsProducerListener = new MicrometerProducerListener<>(bootstrap.meterRegistry());
            KafkaWithMetricsProducerFactory producerFactory = new KafkaWithMetricsProducerFactory(producerProps, metricsProducerListener);

            return new KafkaTemplate<>(producerFactory);
        }
        @Bean
        public BatchConsumerSingleRecord batchConsumerSingleRecord(EbeanFactoryBean factory, KafkaTemplate<byte[], byte[]> kafkaTemplate) throws Exception {
            return new BatchConsumerSingleRecord(factory, kafkaTemplate);
        }
        @Bean
        public HikariDataSourceFactoryBean ds() {
            return new HikariDataSourceFactoryBean(psi());
        }
        @Bean
        public EbeanDatabaseConfig ebeanConfig(HikariDataSourceFactoryBean factory) throws Exception {
            EbeanDatabaseConfig cfg = new EbeanDatabaseConfig(factory.getObject(), bootstrap.props());

            cfg.setDatabasePlatform(new Postgres9Platform());
            cfg.addClass(Account.class);
            cfg.addClass(AccountBalance.class);
            cfg.addClass(AccountBalanceId.class);

            return cfg;
        }
        @Bean
        public EbeanFactoryBean ebean(EbeanDatabaseConfig config) throws Exception {
            return new EbeanFactoryBean(config) {
                @Override
                protected JpaManager createInstance() throws Exception {
                    EbeanJpaManager ebean = (EbeanJpaManager) super.createInstance();
                    FlywayUberRunner.run(ebean, SCHEMA_CORE);
                    return ebean;
                }
            };
        }
        @Bean
        public SmartLifecycle channel(ChangeConsumer<ChangeEvent<SourceRecord, SourceRecord>> consumer) {
            return new SmartLifecycle() {
                private DebeziumEngine<ChangeEvent<SourceRecord, SourceRecord>> engine;
                private boolean running;

                @Override
                public boolean isAutoStartup() {
                    return false;
                }
                @Override
                public void start() {
                    String db = psi().getPath();
                    String schema = "core";
                    int kafkaPort = bootstrap.cfg().getInteger("kafka.port");
                    String tbl1 = schema + "." + TBL_ACCOUNT;
                    String tbl2 = schema + "." + TBL_ACCOUNT_BALANCE;

                    Map<String, String> map = Maps.newHashMap();

                    map.put(EmbeddedEngine.ENGINE_NAME.name(), "default");
                    map.put(EmbeddedEngine.CONNECTOR_CLASS.name(), PostgresConnector.class.getName());

                    map.put(RelationalDatabaseConnectorConfig.SERVER_NAME.name(), "dev");
                    map.put(RelationalDatabaseConnectorConfig.HOSTNAME.name(), psi().getHost());
                    map.put(RelationalDatabaseConnectorConfig.PORT.name(), Integer.toString(psi().getPort()));
                    map.put(RelationalDatabaseConnectorConfig.USER.name(), psi().getUserName());
                    map.put(RelationalDatabaseConnectorConfig.PASSWORD.name(), psi().getPassword());
                    map.put(RelationalDatabaseConnectorConfig.DATABASE_NAME.name(), db);
                    map.put(RelationalDatabaseConnectorConfig.SCHEMA_INCLUDE_LIST.name(), SCHEMA_CORE);
                    map.put(RelationalDatabaseConnectorConfig.TABLE_INCLUDE_LIST.name(), Joiner.on(',').join(ImmutableList.of(tbl1, tbl2)));

                    map.put(PostgresConnectorConfig.PLUGIN_NAME.name(), LogicalDecoder.PGOUTPUT.getValue());
                    map.put(PostgresConnectorConfig.PUBLICATION_AUTOCREATE_MODE.name(), AutoCreateMode.ALL_TABLES.getValue());
                    map.put(PostgresConnectorConfig.SNAPSHOT_MODE.name(), SnapshotMode.INITIAL.getValue());

                    map.put(EmbeddedEngine.OFFSET_STORAGE.name(), KafkaOffsetBackingStore.class.getName());
                    map.put(EmbeddedEngine.OFFSET_STORAGE_KAFKA_TOPIC.name(), KafkaOffsetBackingStore.class.getName());
                    map.put(EmbeddedEngine.OFFSET_STORAGE_KAFKA_PARTITIONS.name(), Integer.toString(1));
                    map.put(EmbeddedEngine.OFFSET_STORAGE_KAFKA_REPLICATION_FACTOR.name(), Integer.toString(1));

                    map.put(DistributedConfig.OFFSET_STORAGE_TOPIC_CONFIG, "debezium-offset");
                    map.put(WorkerConfig.TOPIC_CREATION_ENABLE_CONFIG, Boolean.toString(true));
                    map.put(WorkerConfig.BOOTSTRAP_SERVERS_CONFIG, "localhost:" + kafkaPort);

                    map.put(HistorizedRelationalDatabaseConnectorConfig.DATABASE_HISTORY.name(), KafkaDatabaseHistory.class.getName());

                    map.put(DatabaseHistory.SKIP_UNPARSEABLE_DDL_STATEMENTS.name(), Boolean.toString(false));
                    map.put(DatabaseHistory.STORE_ONLY_CAPTURED_TABLES_DDL.name(), Boolean.toString(false));

                    Properties props = new Properties();
                    props.putAll(map);

                    engine = DebeziumEngine.create(Connect.class).using(props).notifying(consumer).build();

                    bootstrap.globalPlatform().execute(engine);

                    this.running = true;
                }
                @Override
                public void stop() {
                    if (Objects.nonNull(engine)) {
                        try {
                            engine.close();
                        } catch (IOException err) {
                            logger.error(err.getMessage(), err);
                        }
                    }
                    this.running = false;
                }
                @Override
                public boolean isRunning() {
                    return running;
                }
            };
        }
    }
}
