package com.turbospaces.logging;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;

import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import org.awaitility.Awaitility;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.slf4j.LoggerFactory;
import org.springframework.test.util.ReflectionTestUtils;

import com.google.common.math.DoubleMath;
import com.turbospaces.boot.MockCloud;
import com.turbospaces.boot.SimpleBootstrap;
import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.ApplicationProperties;

import ch.qos.logback.classic.Logger;
import ch.qos.logback.classic.LoggerContext;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.Appender;
import io.github.resilience4j.ratelimiter.RateLimiterConfig;
import io.github.resilience4j.ratelimiter.RateLimiterRegistry;
import io.micrometer.core.instrument.composite.CompositeMeterRegistry;
import io.micrometer.core.instrument.search.MeterNotFoundException;
import io.sentry.SentryClient;
import io.sentry.event.EventBuilder;

public class SentryAppenderContextTest {

    private final double doubleTolerance = 0.5;

    @Test
    void baseScenarioCheckRateLimiterAndMetrics_success() throws Throwable {

        Map<String, Object> propMap = new HashMap<>();
        propMap.put("app.dev.mode", false);
        propMap.put(Logback.LOGBACK_RATE_LIMITER_SENTRY_APPENDER_ENABLED, true);
        propMap.put(Logback.LOGBACK_RATE_LIMITER_SENTRY_APPENDER_METRICS_ENABLED, true);

        ApplicationProperties applicationProperties = SentryAppenderContextTest.getApplicationProperties(propMap);

        org.slf4j.Logger logger = LoggerFactory.getLogger(this.getClass());
        SimpleBootstrap bootstrap = new SimpleBootstrap(applicationProperties);

        SentryClient sentryClientMock = mockSentryClient();
        Integer logbackRateLimiterSentryAppenderCount = applicationProperties.LOGBACK_RATE_LIMITER_SENTRY_APPENDER_COUNT.get();
        int amountOfErrors = logbackRateLimiterSentryAppenderCount * 3;
        String errorMessage = "n errors call: {}";

        for (int i = 0; i < amountOfErrors; i++){
            logger.error(errorMessage, i);
        }

        CompositeMeterRegistry meterRegistry = getCompositeMeterRegistry();

        Awaitility.await()
                .atMost(5000, TimeUnit.MILLISECONDS)
                .pollDelay(500, TimeUnit.MILLISECONDS)
                .until(() -> {
                    Double actual = meterRegistry.get("logback.sentry-appender.skipped-events.counter").counter().count();
                    Double expected = Double.valueOf(amountOfErrors - logbackRateLimiterSentryAppenderCount);

                    return DoubleMath.fuzzyEquals(actual, expected, doubleTolerance);
                });

        Assertions.assertTrue(meterRegistry == bootstrap.meterRegistry());
        double amountOfSkippedErrors = meterRegistry.get("logback.sentry-appender.skipped-events.counter").counter().count();
        Assertions.assertTrue(DoubleMath.fuzzyEquals(amountOfErrors - logbackRateLimiterSentryAppenderCount, amountOfSkippedErrors, doubleTolerance));

        RateLimiterRegistry rateLimiterRegistry = getRateLimiterRegistry();
        Assertions.assertNotEquals(bootstrap.rateLimiterRegistry(), rateLimiterRegistry);

        String rateLimiterKey = "rate-limiter-sentry-appender" + "." + this.getClass().getName();
        boolean containsRateLimiter = rateLimiterRegistry.getAllRateLimiters().stream().anyMatch(rlim -> rlim.getName().equals(rateLimiterKey));
        Assertions.assertTrue(containsRateLimiter, "RateLimiter does not contains limiter from SentryAppender: " + rateLimiterKey);

        Optional<RateLimiterConfig> currentRateLimiterConfig = rateLimiterRegistry.getConfiguration(rateLimiterKey);
        boolean limitExceeded = !rateLimiterRegistry.rateLimiter(rateLimiterKey, String.valueOf(currentRateLimiterConfig)).acquirePermission();

        Assertions.assertTrue(limitExceeded, "Failed checked rate limitExceeded with period:  " + applicationProperties.LOGBACK_RATE_LIMITER_SENTRY_APPENDER_PERIOD.get());

        verify(sentryClientMock, times(logbackRateLimiterSentryAppenderCount)).sendEvent(any(EventBuilder.class));

    }

    @Test
    void baseScenarioCheckRateLimiterAndMetricsWithExceptionInLogger_success() throws Throwable {

        Map<String, Object> propMap = new HashMap<>();
        propMap.put("app.dev.mode", false);
        propMap.put(Logback.LOGBACK_RATE_LIMITER_SENTRY_APPENDER_ENABLED, true);
        propMap.put(Logback.LOGBACK_RATE_LIMITER_SENTRY_APPENDER_METRICS_ENABLED, true);

        ApplicationProperties applicationProperties = SentryAppenderContextTest.getApplicationProperties(propMap);

        org.slf4j.Logger logger = LoggerFactory.getLogger(this.getClass());
        SimpleBootstrap bootstrap = new SimpleBootstrap(applicationProperties);

        SentryClient sentryClientMock = mockSentryClient();
        Integer logbackRateLimiterSentryAppenderCount = applicationProperties.LOGBACK_RATE_LIMITER_SENTRY_APPENDER_COUNT.get();
        int amountOfErrors = logbackRateLimiterSentryAppenderCount * 3;
        String errorMessage = "n errors call: {}";

        for (int i = 0; i < amountOfErrors; i++){
            logger.error(errorMessage, new IllegalAccessException());
        }

        CompositeMeterRegistry meterRegistry = getCompositeMeterRegistry();

        Awaitility.await()
                .atMost(5000, TimeUnit.MILLISECONDS)
                .pollDelay(500, TimeUnit.MILLISECONDS)
                .until(() -> {
                    Double actual = meterRegistry.get("logback.sentry-appender.skipped-events.counter").counter().count();
                    Double expected = Double.valueOf(amountOfErrors - logbackRateLimiterSentryAppenderCount);

                    return DoubleMath.fuzzyEquals(actual, expected, doubleTolerance);
                });

        Assertions.assertTrue(meterRegistry == bootstrap.meterRegistry());
        double amountOfSkippedErrors = meterRegistry.get("logback.sentry-appender.skipped-events.counter").counter().count();
        Assertions.assertTrue(DoubleMath.fuzzyEquals(amountOfErrors - logbackRateLimiterSentryAppenderCount, amountOfSkippedErrors, doubleTolerance));

        RateLimiterRegistry rateLimiterRegistry = getRateLimiterRegistry();
        Assertions.assertNotEquals(bootstrap.rateLimiterRegistry(), rateLimiterRegistry);

        String rateLimiterKey = "rate-limiter-sentry-appender" + "." + this.getClass().getName() + "_" + "IllegalAccessException";
        boolean containsRateLimiter = rateLimiterRegistry.getAllRateLimiters().stream().anyMatch(rlim -> rlim.getName().equals(rateLimiterKey));
        Assertions.assertTrue(containsRateLimiter, "RateLimiter does not contains limiter from SentryAppender: " + rateLimiterKey);

        Optional<RateLimiterConfig> currentRateLimiterConfig = rateLimiterRegistry.getConfiguration(rateLimiterKey);
        boolean limitExceeded = !rateLimiterRegistry.rateLimiter(rateLimiterKey, String.valueOf(currentRateLimiterConfig)).acquirePermission();

        Assertions.assertTrue(limitExceeded, "Failed checked rate limitExceeded with period:  " + applicationProperties.LOGBACK_RATE_LIMITER_SENTRY_APPENDER_PERIOD.get());

        verify(sentryClientMock, times(logbackRateLimiterSentryAppenderCount)).sendEvent(any(EventBuilder.class));

    }

    @Test
    void scenarioCheckRateLimiterAndMetricsWithRepeatedSendToSentry() throws Throwable {

        long periodInMilliseconds = 500;

        Map<String, Object> propMap = new HashMap<>();
        propMap.put("app.dev.mode", false);
        propMap.put(Logback.LOGBACK_RATE_LIMITER_SENTRY_APPENDER_PERIOD, Duration.ofMillis(periodInMilliseconds));
        propMap.put(Logback.LOGBACK_RATE_LIMITER_SENTRY_APPENDER_COUNT, 10);
        propMap.put(Logback.LOGBACK_RATE_LIMITER_SENTRY_APPENDER_ENABLED, true);
        propMap.put(Logback.LOGBACK_RATE_LIMITER_SENTRY_APPENDER_METRICS_ENABLED, true);

        ApplicationProperties applicationProperties = SentryAppenderContextTest.getApplicationProperties(propMap);

        org.slf4j.Logger logger = LoggerFactory.getLogger(this.getClass());

        SimpleBootstrap bootstrap = new SimpleBootstrap(applicationProperties);

        SentryClient sentryClientMock = mockSentryClient();

        Integer logbackRateLimiterSentryAppenderCount = applicationProperties.LOGBACK_RATE_LIMITER_SENTRY_APPENDER_COUNT.get();
        int amountOfErrors = logbackRateLimiterSentryAppenderCount * 3;
        String errorMessage = "n errors call: {}";

        for (int i = 0; i < amountOfErrors; i++){
            logger.error(errorMessage, i);
        }

        CompositeMeterRegistry meterRegistry = getCompositeMeterRegistry();

        Awaitility.await()
                .atMost(5000, TimeUnit.MILLISECONDS)
                .pollDelay(500, TimeUnit.MILLISECONDS)
                .until(() -> {
                    Double actual = meterRegistry.get("logback.sentry-appender.skipped-events.counter").counter().count();
                    Double expected = Double.valueOf(amountOfErrors - logbackRateLimiterSentryAppenderCount);

                    return DoubleMath.fuzzyEquals(actual, expected, doubleTolerance);
                });

        for (int i = 0; i < amountOfErrors; i++){
            logger.error(errorMessage, i);
        }

        Awaitility.await()
                .atMost(5000, TimeUnit.MILLISECONDS)
                .pollDelay(500, TimeUnit.MILLISECONDS)
                .until(() -> {
                    Double actual = meterRegistry.get("logback.sentry-appender.skipped-events.counter").counter().count();
                    Double expected =  2*Double.valueOf(amountOfErrors - logbackRateLimiterSentryAppenderCount);

                    return DoubleMath.fuzzyEquals(actual, expected, doubleTolerance);
                });

        double amountOfSkippedErrors = meterRegistry.get("logback.sentry-appender.skipped-events.counter").counter().count();
        Assertions.assertTrue(DoubleMath.fuzzyEquals(amountOfSkippedErrors, 2 * (amountOfErrors - logbackRateLimiterSentryAppenderCount), doubleTolerance ) );

        RateLimiterRegistry rateLimiterRegistry = getRateLimiterRegistry();
        String rateLimiterKey = "rate-limiter-sentry-appender" + "." + this.getClass().getName();
        boolean containsRateLimiter = rateLimiterRegistry.getAllRateLimiters().stream().anyMatch(rlim -> rlim.getName().equals(rateLimiterKey));
        Assertions.assertTrue(containsRateLimiter, "RateLimiter don't contains limiter from SentryAppender: " + rateLimiterKey);

        verify(sentryClientMock, times(2 * logbackRateLimiterSentryAppenderCount)).sendEvent(any(EventBuilder.class));

    }

    @Test
    void scenarioWithDisabledRateLimiter() throws Throwable {

        Map<String, Object> propMap = new HashMap<>();
        propMap.put("app.dev.mode", false);

        ApplicationProperties applicationProperties = SentryAppenderContextTest.getApplicationProperties(propMap);

        org.slf4j.Logger logger = LoggerFactory.getLogger(this.getClass());

        SimpleBootstrap bootstrap = new SimpleBootstrap(applicationProperties);

        SentryClient sentryClientMock = mockSentryClient();

        Integer logbackRateLimiterSentryAppenderCount = applicationProperties.LOGBACK_RATE_LIMITER_SENTRY_APPENDER_COUNT.get();
        int amountOfErrors = logbackRateLimiterSentryAppenderCount * 3;
        String errorMessage = "n errors call: {}";

        for (int i = 0; i < amountOfErrors; i++){
            logger.error(errorMessage, i);
        }
        Thread.sleep(500);

        CompositeMeterRegistry meterRegistry = getCompositeMeterRegistry();
        Assertions.assertThrows(MeterNotFoundException.class, () -> meterRegistry.get("logback.sentry-appender.skipped-events.counter").counter());

        RateLimiterRegistry rateLimiterRegistry = getRateLimiterRegistry();
        String rateLimiterKey = "rate-limiter-sentry-appender" + "." + this.getClass().getName();
        boolean containsRateLimiter = rateLimiterRegistry.getAllRateLimiters().stream().anyMatch(rlim -> rlim.getName().equals(rateLimiterKey));
        Assertions.assertFalse(containsRateLimiter, "RateLimiter contains limiter from SentryAppender: " + rateLimiterKey);

        verify(sentryClientMock, times(amountOfErrors)).sendEvent(any(EventBuilder.class));

    }

    private static ApplicationProperties getApplicationProperties(Map<String, Object> setDefaultProp) throws Exception {

        ApplicationConfig cfg = MockCloud.newMock().build();
        cfg.loadLocalDevProperties();
        setDefaultProp.entrySet().forEach(prop ->{
            cfg.setDefaultProperty(prop.getKey(), prop.getValue());
        });

        return new ApplicationProperties(cfg);
    }

    private static CompositeMeterRegistry getCompositeMeterRegistry() {
        return (CompositeMeterRegistry) ((LoggerContext) LoggerFactory.getILoggerFactory()).getObject("metrics-registry");
    }

    private static RateLimiterRegistry getRateLimiterRegistry() {
        LoggerContext loggerContext = (LoggerContext) LoggerFactory.getILoggerFactory();
        Appender<ILoggingEvent> sentryAppender = null;
        for (Logger log : loggerContext.getLoggerList()) {
            Appender<ILoggingEvent> appender = log.getAppender("SENTRY");
            if (appender != null){
                sentryAppender = appender;
                break;
            }
        }
        return (RateLimiterRegistry) ReflectionTestUtils.getField(sentryAppender, "rateLimiterRegistry");
    }

    private static SentryClient mockSentryClient() {
        LoggerContext loggerContext = (LoggerContext) LoggerFactory.getILoggerFactory();
        Appender<ILoggingEvent> sentryAppender = null;
        for (Logger log : loggerContext.getLoggerList()) {
            Appender<ILoggingEvent> appender = log.getAppender("SENTRY");
            if (appender != null){
                sentryAppender = appender;
                break;
            }
        }
        SentryClient sentryClientMock = mock(SentryClient.class);
        ReflectionTestUtils.setField(sentryAppender, "sentry", sentryClientMock);
        doNothing().when(sentryClientMock).sendEvent(any(EventBuilder.class));
        return sentryClientMock;
    }

}