package com.turbospaces.rpc;

import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import com.google.protobuf.Message;
import com.turbospaces.api.Topic;
import com.turbospaces.api.facade.NotificationWrapperFacade;
import com.turbospaces.dispatch.EventQueuePostSpec;
import com.turbospaces.dispatch.NotifyQueuePostSpec;
import com.turbospaces.dispatch.RequestQueuePostSpec;
import com.turbospaces.dispatch.TransactionalRequestOutcome;

import io.netty.util.AsciiString;
import lombok.RequiredArgsConstructor;

@RequiredArgsConstructor
public class TransitionalPostTemplate implements QueuePostTemplate<String> {

    private final QueuePostTemplate<?> kafka;
    private final QueuePostTemplate<String> pubsub;

    @Override
    public CompletableFuture sendEvent(EventQueuePostSpec spec) {
        kafka.sendEvent(spec);
        return pubsub.sendEvent(spec);
    }

    @Override
    public CompletableFuture sendEvent(Topic destination, Message payload) {
        kafka.sendEvent(destination, payload);
        return pubsub.sendEvent(destination, payload);
    }

    @Override
    public CompletableFuture sendEvent(Topic destination, Message payload, AsciiString routingKey) {
        kafka.sendEvent(destination, payload, routingKey);
        return pubsub.sendEvent(destination, payload, routingKey);
    }

    @Override
    public CompletableFuture sendNotify(NotifyQueuePostSpec spec) {
        kafka.sendNotify(spec);
        return pubsub.sendNotify(spec);
    }

    @Override
    public CompletableFuture sendNotify(Topic destination, NotificationWrapperFacade payload, AsciiString routingKey) {
        kafka.sendNotify(destination, payload, routingKey);
        return pubsub.sendNotify(destination, payload, routingKey);
    }

    @Override
    public void publishNotifications(Topic topic, TransactionalRequestOutcome outcome) {
        kafka.publishNotifications(topic, outcome);
        pubsub.publishNotifications(topic, outcome);
    }

    @Override
    public void publishEvents(Topic topic, TransactionalRequestOutcome outcome) {
        kafka.publishEvents(topic, outcome);
        pubsub.publishEvents(topic, outcome);
    }

    @Override
    public void publishReply(TransactionalRequestOutcome outcome) {
        pubsub.publishReply(outcome);
    }

    @Override
    public WrappedQueuePost sendReq(RequestQueuePostSpec spec) {
        return pubsub.sendReq(spec);
    }

    @Override
    public WrappedQueuePost sendReq(Topic destination, Topic replyTo, Message payload, AsciiString routingKey) {
        return pubsub.sendReq(destination, replyTo, payload, routingKey);
    }

    @Override
    public WrappedQueuePost sendReq(Topic destination, Topic replyTo, Message payload, AsciiString routingKey, Supplier<Duration> timeout) {
        return pubsub.sendReq(destination, replyTo, payload, routingKey);
    }

}
