package com.turbospaces.rpc;

import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.RandomStringUtils;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.mockito.ArgumentMatchers;
import org.mockito.Mockito;

import com.google.protobuf.Any;
import com.google.protobuf.Timestamp;
import com.turbospaces.api.facade.HeadersFacade;
import com.turbospaces.api.facade.RequestWrapperFacade;
import com.turbospaces.api.facade.ResponseStatusFacade;
import com.turbospaces.api.facade.ResponseWrapperFacade;
import com.turbospaces.boot.MockCloud;
import com.turbospaces.boot.SimpleBootstrap;
import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.dispatch.EmbeddedTransactionalRequest;
import com.turbospaces.dispatch.SafeRequestHandler;
import com.turbospaces.dispatch.TransactionalRequest;
import com.turbospaces.dispatch.TransactionalRequestHandler;
import com.turbospaces.dispatch.TransactionalRequestOutcome;
import com.turbospaces.dispatch.WorkerCompletableTask;
import com.turbospaces.executor.WorkUnit;

import api.v1.ApiFactory;
import io.micrometer.core.instrument.Measurement;
import io.micrometer.core.instrument.Timer;
import io.micrometer.core.instrument.search.RequiredSearch;
import io.opentracing.noop.NoopSpan;
import io.vavr.CheckedRunnable;
import lombok.extern.slf4j.Slf4j;

@Slf4j
class SafeRequestHandlerTest {
    CountDownLatch latch = new CountDownLatch(1);
    String key = RandomStringUtils.randomAlphanumeric(4);
    WorkUnit unit = new WorkUnit() {
        @Override
        public String topic() {
            return "junit";
        }
        @Override
        public long timestamp() {
            return System.currentTimeMillis();
        }
        @Override
        public byte[] key() {
            return key.getBytes();
        }
    };

    @Test
    void works() throws Throwable {
        ApplicationConfig cfg = MockCloud.newMock().build();
        ApplicationProperties props = new ApplicationProperties(cfg);
        SimpleBootstrap boot = new SimpleBootstrap(props);
        boot.run();

        try {
            var req = Timestamp.newBuilder().setSeconds(TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis())).build();

            var apiFactory = Mockito.mock(ApiFactory.class);

            var headersw = Mockito.mock(HeadersFacade.class);
            Mockito.when(headersw.getMessageId()).thenReturn(UUID.randomUUID().toString());

            var statusw = Mockito.mock(ResponseStatusFacade.class);
            Mockito.when(statusw.isOK()).thenReturn(true);

            var respw = Mockito.mock(ResponseWrapperFacade.class);
            Mockito.when(respw.status()).thenReturn(statusw);

            var reqw = Mockito.mock(RequestWrapperFacade.class);
            Mockito.when(reqw.headers()).thenReturn(headersw);
            Mockito.when(reqw.body()).thenReturn(Any.pack(req));
            Mockito.when(reqw.toReply(ArgumentMatchers.any())).thenReturn(respw);

            Semaphore semaphore = new Semaphore(0);

            CheckedRunnable task1 = Mockito.mock(CheckedRunnable.class);
            CheckedRunnable task2 = Mockito.mock(CheckedRunnable.class);

            var handler = new SafeRequestHandler<>(
                    reqw,
                    apiFactory,
                    NoopSpan.INSTANCE,
                    unit,
                    new EmbeddedTransactionalRequest<>(Timestamp.class, Timestamp.newBuilder(), unit, reqw, latch),
                    new TransactionalRequestHandler<>() {
                        @Override
                        public boolean actorIsRequired() {
                            return false;
                        }

                        @Override
                        public boolean isImmediateAcknowledge() {
                            return false;
                        }

                        @Override
                        public void apply(TransactionalRequest<Timestamp, Timestamp.Builder> cmd) throws Exception {
                            cmd.addMetricTag("provider", Long.toString(System.currentTimeMillis()));

                            cmd.replyWhen(boot.globalPlatform().submit((CheckedRunnable) () -> {
                                semaphore.acquire();
                                task1.run();
                            }));

                            cmd.replyWhen(boot.globalPlatform().submit((CheckedRunnable) () -> {
                                semaphore.acquire();
                                task2.run();
                            }));
                        }
                    });

            handler.setBootstrap(boot);
            WorkerCompletableTask task = handler.get();
            handler.run();

            Mockito.verify(task1, Mockito.never()).run();
            Mockito.verify(task2, Mockito.never()).run();

            semaphore.release(2);

            TransactionalRequestOutcome operationOutcome = task.get();

            Mockito.verify(task1).run();
            Mockito.verify(task2).run();

            RequiredSearch search = boot.meterRegistry().get(SafeRequestHandler.METRIC_NAME);
            Timer timer = search.timer();
            Measurement next = timer.measure().iterator().next();
            log.info(next.toString());

            Assertions.assertTrue(operationOutcome.getNotifications().isEmpty());
            Assertions.assertTrue(operationOutcome.getEventStream().isEmpty());
        } finally {
            boot.shutdown();
        }
    }
    @Test
    void applicationExceptionInMain() throws Throwable {
        ApplicationConfig cfg = MockCloud.newMock().build();
        ApplicationProperties props = new ApplicationProperties(cfg);
        SimpleBootstrap boot = new SimpleBootstrap(props);
        boot.run();

        try {
            var req = Timestamp.newBuilder().setSeconds(TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis())).build();

            var apiFactory = Mockito.mock(ApiFactory.class);

            var headersw = Mockito.mock(HeadersFacade.class);
            Mockito.when(headersw.getMessageId()).thenReturn(UUID.randomUUID().toString());

            var statusw = Mockito.mock(ResponseStatusFacade.class);
            Mockito.when(statusw.isNotFound()).thenReturn(true);

            var respw = Mockito.mock(ResponseWrapperFacade.class);
            Mockito.when(respw.status()).thenReturn(statusw);

            var reqw = Mockito.mock(RequestWrapperFacade.class);
            Mockito.when(reqw.headers()).thenReturn(headersw);
            Mockito.when(reqw.body()).thenReturn(Any.pack(req));
            Mockito.when(reqw.toExceptionalReply(ArgumentMatchers.any(), ArgumentMatchers.any())).thenReturn(respw);

            var handler = new SafeRequestHandler<Timestamp, Timestamp.Builder>(
                    reqw,
                    apiFactory,
                    NoopSpan.INSTANCE,
                    unit,
                    new EmbeddedTransactionalRequest<Timestamp, Timestamp.Builder>(Timestamp.class, Timestamp.newBuilder(), unit, reqw, latch),
                    new TransactionalRequestHandler<Timestamp, Timestamp.Builder>() {
                        @Override
                        public boolean actorIsRequired() {
                            return false;
                        }
                        @Override
                        public boolean isImmediateAcknowledge() {
                            return false;
                        }
                        @Override
                        public void apply(TransactionalRequest<Timestamp, Timestamp.Builder> cmd) throws Exception {
                            throw new IllegalArgumentException();
                        }
                    });

            handler.setBootstrap(boot);
            WorkerCompletableTask task = handler.get();
            handler.run();
            TransactionalRequestOutcome operationOutcome = task.get();

            RequiredSearch search = boot.meterRegistry().get(SafeRequestHandler.METRIC_NAME);
            Timer timer = search.timer();
            Measurement next = timer.measure().iterator().next();
            log.info(next.toString());

            Assertions.assertTrue(operationOutcome.getReply().status().isNotFound());
        } finally {
            boot.shutdown();
        }
    }
    @Test
    void exceptionInReplyWhen() throws Throwable {
        ApplicationConfig cfg = MockCloud.newMock().build();
        ApplicationProperties props = new ApplicationProperties(cfg);
        SimpleBootstrap boot = new SimpleBootstrap(props);
        boot.run();

        try {
            var req = Timestamp.newBuilder().setSeconds(TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis())).build();

            var apiFactory = Mockito.mock(ApiFactory.class);

            var headersw = Mockito.mock(HeadersFacade.class);
            Mockito.when(headersw.getMessageId()).thenReturn(UUID.randomUUID().toString());

            var statusw = Mockito.mock(ResponseStatusFacade.class);
            Mockito.when(statusw.isSystem()).thenReturn(true);

            var respw = Mockito.mock(ResponseWrapperFacade.class);
            Mockito.when(respw.status()).thenReturn(statusw);

            var reqw = Mockito.mock(RequestWrapperFacade.class);
            Mockito.when(reqw.headers()).thenReturn(headersw);
            Mockito.when(reqw.body()).thenReturn(Any.pack(req));
            Mockito.when(reqw.toExceptionalReply(ArgumentMatchers.any(), ArgumentMatchers.any())).thenReturn(respw);

            var handler = new SafeRequestHandler<Timestamp, Timestamp.Builder>(
                    reqw,
                    apiFactory,
                    NoopSpan.INSTANCE,
                    unit,
                    new EmbeddedTransactionalRequest<Timestamp, Timestamp.Builder>(Timestamp.class, Timestamp.newBuilder(), unit, reqw, latch),
                    new TransactionalRequestHandler<Timestamp, Timestamp.Builder>() {
                        @Override
                        public boolean actorIsRequired() {
                            return true;
                        }
                        @Override
                        public boolean isImmediateAcknowledge() {
                            return false;
                        }
                        @Override
                        public void apply(TransactionalRequest<Timestamp, Timestamp.Builder> cmd) throws Exception {
                            cmd.replyWhen(boot
                                    .globalPlatform()
                                    .submit(new Callable<Timestamp>() {
                                        @Override
                                        public Timestamp call() throws Exception {
                                            throw new RuntimeException("unable to complete");
                                        }
                                    }));

                        }
                    });

            handler.setBootstrap(boot);
            WorkerCompletableTask task = handler.get();
            handler.run();
            TransactionalRequestOutcome operationOutcome = task.get();

            Assertions.assertTrue(operationOutcome.getNotifications().isEmpty());
            Assertions.assertTrue(operationOutcome.getReply().status().isSystem());
        } finally {
            boot.shutdown();
        }
    }
}
