package com.turbospaces.dispatch;

import java.time.Duration;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import com.turbospaces.api.facade.RequestWrapperFacade;
import com.turbospaces.api.jpa.CompositeStackTracer;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.common.PlatformUtil;
import com.turbospaces.executor.WorkUnit;
import com.turbospaces.mdc.MdcUtil;

import api.v1.ApiFactory;

public abstract class AbstractServerRequestConsumer implements ServiceCallAsyncDispatcher {
    protected final Logger logger = LoggerFactory.getLogger(getClass());
    protected final ApplicationProperties props;
    protected final ApiFactory apiFactory;
    protected final CompositeStackTracer stackTracer;
    protected Duration timeout;

    protected AbstractServerRequestConsumer(
            ApplicationProperties props,
            ApiFactory apiFactory,
            CompositeStackTracer stackTracer) {
        this.props = Objects.requireNonNull(props);
        this.apiFactory = Objects.requireNonNull(apiFactory);
        this.stackTracer = Objects.requireNonNull(stackTracer);
        this.timeout = props.BATCH_COMPLETION_TIMEOUT.get();
    }
    protected Optional<ListenableFuture<TransactionalRequestOutcome>> logAndAccept(
            RequestWrapperFacade reqw,
            WorkUnit unit,
            CountDownLatch latch) throws Throwable {
        String typeUrl = reqw.body().getTypeUrl();

        //
        // ~ set corresponding MDC values
        //
        String operation = PlatformUtil.toLowerUnderscore(typeUrl);
        MdcUtil.setMdc(unit, operation, reqw.headers());

        try {
            logger.info("IN ::: ({}) workUnit: {}", typeUrl, unit);

            //
            // ~ maybe time-outed already
            //
            int origin = reqw.headers().getTimeout();
            boolean outdated = false;
            if (origin > 0) {
                long delta = System.currentTimeMillis() - unit.timestamp();
                if (delta > TimeUnit.SECONDS.toMillis(origin)) {
                    outdated = true;
                }
            }

            if (outdated) {
                latch.countDown();
            } else {
                return Optional.ofNullable(schedule(unit, reqw, latch));
            }
        } catch (Exception err) {
            logger.error(err.getMessage(), err);
            throw err;
        } finally {
            MdcUtil.clearMdc(unit);
        }

        return Optional.empty();
    }
    protected WorkerCompletableTask convertUnhandledException(WorkUnit unit, Throwable err) {
        var respw = apiFactory.responseMapper().toTimeoutReplyWithoutBody(err);

        var delegate = SettableFuture.<TransactionalRequestOutcome> create();
        var completable = new WorkerCompletableTask(delegate, unit, respw.headers().getMessageId());
        var outcome = TransactionalRequestOutcome.builder().reply(respw);
        unit.ifKeyPresent(outcome::key);
        delegate.set(outcome.build());
        return completable;
    }
}
