package com.turbospaces.dispatch;

import java.nio.file.Paths;
import java.util.Date;
import java.util.concurrent.CountDownLatch;

import org.apache.commons.lang3.concurrent.ConcurrentException;
import org.apache.commons.lang3.concurrent.LazyInitializer;

import com.google.common.io.ByteSource;
import com.google.protobuf.Any;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.Message;
import com.turbospaces.api.facade.DefaultRequestWrapperFacade;
import com.turbospaces.api.facade.RequestWrapperFacade;
import com.turbospaces.common.PlatformUtil;
import com.turbospaces.executor.WorkUnit;

import api.v1.ApiFactory;
import api.v1.ObfuscatePrinter;
import api.v1.RequestWrapper;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class EmbeddedTransactionalRequest<REQ extends Message, RESP extends Message.Builder> extends AbstractTransactionalRequest<REQ, RESP> {
    private final Date when;

    public EmbeddedTransactionalRequest(Class<REQ> reqType, RESP prototype, WorkUnit record, RequestWrapperFacade wrapper, CountDownLatch latch) {
        super(prototype, record, wrapper, latch);
        this.req = new LazyInitializer<REQ>() {
            @Override
            protected REQ initialize() throws ConcurrentException {
                Any body = wrapper.body();
                try {
                    REQ unpack = body.unpack(reqType);
                    log.debug("unpack: {}", ObfuscatePrinter.shortDebugString(unpack.toBuilder()));
                    return unpack;
                } catch (InvalidProtocolBufferException err) {
                    throw new ConcurrentException(err);
                }
            }
        };
        this.when = new Date();
    }
    @Override
    public REQ request() throws Exception {
        return req.get();
    }
    @Override
    public Date timestamp() {
        return when;
    }
    public static <T> String jdbcOp(T req, String... args) {
        String typeUrl = PlatformUtil.toLowerUnderscore(req.getClass().getSimpleName());
        return Paths.get(String.format("jdbc_%s", typeUrl), args).toString();
    }
    @SuppressWarnings("unchecked")
    public static <REQ extends Message, RESP extends Message.Builder> EmbeddedTransactionalRequest<REQ, RESP> mock(
            ApiFactory apiFactory,
            REQ req,
            RESP prototype) {
        Class<REQ> type = (Class<REQ>) req.getClass();

        //
        // ~ request wrapper
        //
        RequestWrapper.Builder reqwb = RequestWrapper.newBuilder();
        reqwb.setHeaders(api.v1.Headers.newBuilder().setMessageId(PlatformUtil.randomUUID().toString()));
        reqwb.setBody(Any.pack(req));
        RequestWrapper reqw = reqwb.build();

        return new EmbeddedTransactionalRequest<REQ, RESP>(
                type,
                prototype,
                new WorkUnit() {
                    @Override
                    public ByteSource value() {
                        return ByteSource.wrap(reqw.toByteArray());
                    }
                    @Override
                    public String topic() {
                        return "any";
                    }
                    @Override
                    public long timestamp() {
                        return System.currentTimeMillis();
                    }
                    @Override
                    public byte[] key() {
                        return null;
                    }
                },
                new DefaultRequestWrapperFacade(apiFactory.eventTemplate(), reqw),
                new CountDownLatch(1));
    }
}
