package com.turbospaces.executor;

import java.time.Duration;
import java.util.Collections;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor.AbortPolicy;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

import org.apache.commons.lang3.time.StopWatch;
import org.slf4j.MDC;
import org.springframework.beans.factory.BeanNameAware;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.netflix.archaius.api.Property;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.common.PlatformUtil;

import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.binder.jvm.ExecutorServiceMetrics;
import io.vavr.CheckedFunction0;
import io.vavr.CheckedRunnable;
import lombok.experimental.Delegate;
import lombok.extern.slf4j.Slf4j;
import reactor.blockhound.integration.DefaultBlockHoundIntegration;

@Slf4j
public class DefaultPlatformExecutorService
        implements PlatformExecutorService, BeanNameAware, ExecutorService, InitializingBean, DisposableBean {
    private final ThreadLocal<Boolean> flag = DefaultBlockHoundIntegration.FLAG;
    private final ApplicationProperties props;
    private final MeterRegistry meterRegistry;
    private final BlockingQueue<Runnable> queue;
    private final RejectedExecutionHandler rejectPolicy;
    private String beanName;
    private Property<Integer> minPoolSize;
    private Property<Integer> maxPoolSize;
    private Property<Duration> keepAlive;

    @Delegate(types = { ExecutorService.class })
    private ThreadPoolExecutor executor;

    public DefaultPlatformExecutorService(ApplicationProperties props, MeterRegistry meterRegistry) {
        this(props,
                meterRegistry,
                props.APP_PLATFORM_MAX_IDLE,
                props.APP_PLATFORM_MIN_SIZE,
                props.APP_PLATFORM_MAX_SIZE);
    }
    public DefaultPlatformExecutorService(
            ApplicationProperties props,
            MeterRegistry meterRegistry,
            Property<Duration> keepAlive,
            Property<Integer> minPoolSize,
            Property<Integer> maxPoolSize) {
        this.props = Objects.requireNonNull(props);
        this.meterRegistry = Objects.requireNonNull(meterRegistry);
        this.keepAlive = Objects.requireNonNull(keepAlive);
        this.minPoolSize = Objects.requireNonNull(minPoolSize);
        this.maxPoolSize = Objects.requireNonNull(maxPoolSize);
        this.queue = new SynchronousQueue<Runnable>();
        this.rejectPolicy = new AbortPolicy();
    }
    @Override
    public void setBeanName(String name) {
        this.beanName = Objects.requireNonNull(name);

        minPoolSize = props.cfg().factory().get(beanName + "." + "platform.min-size", int.class).orElseGet(minPoolSize.getKey()).orElse(minPoolSize.get());
        maxPoolSize = props.cfg().factory().get(beanName + "." + "platform.max-size", int.class).orElseGet(maxPoolSize.getKey()).orElse(maxPoolSize.get());
        keepAlive = props.cfg().factory().get(beanName + "." + "platform.max-idle", Duration.class).orElseGet(keepAlive.getKey()).orElse(keepAlive.get());
    }
    public int prestartAllCoreThreads() {
        return executor.prestartAllCoreThreads();
    }
    @Override
    public void afterPropertiesSet() {
        //
        // ~ thread factory
        //
        ThreadFactoryBuilder factory = new ThreadFactoryBuilder();
        factory.setDaemon(false);
        factory.setNameFormat(beanName + "-%d");
        factory.setThreadFactory(new ThreadFactory() {
            @Override
            public Thread newThread(Runnable r) {
                return new PlatformThread(props, r);
            }
        });

        //
        // ~ actual executor
        //
        executor = new ThreadPoolExecutor(
                minPoolSize.get(),
                maxPoolSize.get(),
                keepAlive.get().toSeconds(),
                TimeUnit.SECONDS,
                queue,
                factory.build(),
                rejectPolicy);

        log.info("{} has been created min: {}, max: {}, keepAlive: {}", beanName, minPoolSize.get(), maxPoolSize.get(), keepAlive.get());

        minPoolSize.subscribe(new Consumer<Integer>() {
            @Override
            public void accept(Integer newValue) {
                log.info("accepting new corePool value: {}, executor: {}", newValue, beanName);
                executor.setCorePoolSize(newValue);
            }
        });
        maxPoolSize.subscribe(new Consumer<Integer>() {
            @Override
            public void accept(Integer newValue) {
                log.info("accepting new maxPool value: {}, executor: {}", newValue, beanName);
                executor.setMaximumPoolSize(newValue);
            }
        });
        keepAlive.subscribe(new Consumer<Duration>() {
            @Override
            public void accept(Duration newValue) {
                log.info("accepting new keepAlive value: {}, executor: {}", newValue, beanName);
                executor.setKeepAliveTime(newValue.toSeconds(), TimeUnit.SECONDS);
            }
        });

        //
        // ~ bind metrics
        //
        ExecutorServiceMetrics metrics = new ExecutorServiceMetrics(executor, beanName, Collections.emptyList());
        metrics.bindTo(meterRegistry);
    }
    @Override
    public void destroy() {
        StopWatch stopWatch = StopWatch.createStarted();
        Duration timeout = props.APP_PLATFORM_GRACEFUL_SHUTDOWN_TIMEOUT.get();
        PlatformUtil.shutdownExecutor(executor, timeout);
        stopWatch.stop();

        log.debug("stopped executor-service: {} in {}", toString(), stopWatch);
    }
    @Override
    public void execute(Runnable command) {
        Map<String, String> mdcContextMap = MDC.getCopyOfContextMap();
        executor.execute(wrapRunnable(command, mdcContextMap));
    }
    @Override
    public ListenableFuture<?> submit(CheckedRunnable action) {
        //
        // ~ we can afford it, we know that even if we reach max threads, any new task will be rejected
        // ~ so we can safely assume that this is actually not blocking
        //
        boolean toReset = flag.get();
        try {
            flag.set(false);
            return PlatformExecutorService.super.submit(action);
        } finally {
            flag.set(toReset);
        }
    }
    @Override
    public <T> ListenableFuture<T> submit(CheckedFunction0<T> action) {
        //
        // ~ we can afford it, we know that even if we reach max threads, any new task will be rejected
        // ~ so we can safely assume that this is actually not blocking
        //
        boolean toReset = flag.get();
        try {
            flag.set(false);
            return PlatformExecutorService.super.submit(action);
        } finally {
            flag.set(toReset);
        }
    }
    @Override
    public String toString() {
        return String.format("%s(%s)", beanName, executor);
    }
}
