package com.turbospaces.executor;

import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.apache.commons.lang3.time.StopWatch;
import org.slf4j.MDC;
import org.springframework.beans.factory.BeanNameAware;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableScheduledFuture;
import com.google.common.util.concurrent.ListeningScheduledExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.turbospaces.boot.PlatformScheduledExecutorService;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.common.PlatformUtil;

import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.binder.jvm.ExecutorServiceMetrics;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

@Slf4j
@RequiredArgsConstructor
public class DefaultPlatformScheduledExecutorService
        implements PlatformScheduledExecutorService, InitializingBean, DisposableBean, BeanNameAware {
    private final ApplicationProperties props;
    private final MeterRegistry meterRegistry;

    private String beanName = "platform";
    private ListeningScheduledExecutorService executor;

    @Override
    public void setBeanName(String name) {
        this.beanName = Objects.requireNonNull(name);
    }
    @Override
    public void afterPropertiesSet() {
        //
        // ~ thread factory
        //
        ThreadFactoryBuilder factory = new ThreadFactoryBuilder();
        factory.setDaemon(false);
        factory.setNameFormat("global-worker-%d");
        factory.setThreadFactory(new ThreadFactory() {
            @Override
            public Thread newThread(Runnable r) {
                return new PlatformThread(props, r);
            }
        });

        int size = props.APP_PLATFORM_POOL_SIZE.get();
        LogQueueFullCallerRunsPolicy callerRunsPolicy = new LogQueueFullCallerRunsPolicy();
        ThreadFactory threadFactory = factory.build();

        //
        // ~ locally reduce the number of threads to spike potential problems faster
        //
        if (props.APP_DEV_MODE.get()) {
            size = Runtime.getRuntime().availableProcessors();
        }

        //
        // ~ listenable decorator
        //
        executor = MoreExecutors.listeningDecorator(new BlockhoundThreadPoolExecutorWrapper(size, threadFactory, callerRunsPolicy));

        //
        // ~ bind metrics
        //
        ExecutorServiceMetrics metrics = new ExecutorServiceMetrics(executor, beanName, Collections.emptyList());
        metrics.bindTo(meterRegistry);
    }
    @Override
    public void destroy() {
        StopWatch stopWatch = StopWatch.createStarted();
        Duration timeout = props.APP_PLATFORM_GRACEFUL_SHUTDOWN_TIMEOUT.get();
        PlatformUtil.shutdownExecutor(executor, timeout);
        stopWatch.stop();

        log.debug("stopped global scheduled-executor-service: {} in {}", executor, stopWatch);
    }
    @Override
    public void execute(Runnable command) {
        Map<String, String> mdcContextMap = MDC.getCopyOfContextMap();
        executor.execute(wrapRunnable(command, mdcContextMap));
    }
    @Override
    public ListenableFuture<?> submit(Runnable task) {
        Map<String, String> mdcContextMap = MDC.getCopyOfContextMap();
        return executor.submit(wrapRunnable(task, mdcContextMap));
    }
    @Override
    public <T> ListenableFuture<T> submit(Callable<T> task) {
        Map<String, String> mdcContextMap = MDC.getCopyOfContextMap();
        return executor.submit(wrapCallable(task, mdcContextMap));
    }
    @Override
    public <T> ListenableFuture<T> submit(Runnable task, T result) {
        Map<String, String> mdcContextMap = MDC.getCopyOfContextMap();
        return executor.submit(wrapRunnable(task, mdcContextMap), result);
    }
    @Override
    public ListenableScheduledFuture<?> schedule(Runnable command, long delay, TimeUnit unit) {
        Map<String, String> mdcContextMap = MDC.getCopyOfContextMap();
        return executor.schedule(wrapRunnable(command, mdcContextMap), delay, unit);
    }
    public ListenableScheduledFuture<?> schedule(Runnable command, Duration delay) {
        Map<String, String> mdcContextMap = MDC.getCopyOfContextMap();
        return executor.schedule(wrapRunnable(command, mdcContextMap), delay);
    }
    public <V> ListenableScheduledFuture<V> schedule(Callable<V> callable, Duration delay) {
        Map<String, String> mdcContextMap = MDC.getCopyOfContextMap();
        return executor.schedule(wrapCallable(callable, mdcContextMap), delay);
    }
    @Override
    public <V> ListenableScheduledFuture<V> schedule(Callable<V> callable, long delay, TimeUnit unit) {
        Map<String, String> mdcContextMap = MDC.getCopyOfContextMap();
        return executor.schedule(wrapCallable(callable, mdcContextMap), delay, unit);
    }
    @Override
    public ListenableScheduledFuture<?> scheduleAtFixedRate(Runnable command, long initialDelay, long period, TimeUnit unit) {
        return executor.scheduleAtFixedRate(command, initialDelay, period, unit);
    }
    public ListenableScheduledFuture<?> scheduleWithFixedDelay(Runnable command, Duration initialDelay, Duration delay) {
        return executor.scheduleWithFixedDelay(command, initialDelay, delay);
    }
    @Override
    public ScheduledFuture<?> scheduleWithFixedDelay(Runnable command, long initialDelay, long delay, TimeUnit unit) {
        return executor.scheduleWithFixedDelay(command, initialDelay, delay, unit);
    }
    @Override
    public void shutdown() {
        executor.shutdown();
    }
    @Override
    public List<Runnable> shutdownNow() {
        return executor.shutdownNow();
    }
    @Override
    public boolean isShutdown() {
        return executor.isShutdown();
    }
    @Override
    public boolean isTerminated() {
        return executor.isTerminated();
    }
    @Override
    public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
        return executor.awaitTermination(timeout, unit);
    }
    @Override
    public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks) throws InterruptedException {
        return executor.invokeAll(tasks);
    }
    @Override
    public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit) throws InterruptedException {
        return executor.invokeAll(tasks, timeout, unit);
    }
    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks) throws InterruptedException, ExecutionException {
        return executor.invokeAny(tasks);
    }
    @Override
    public <T> T invokeAny(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException {
        return executor.invokeAny(tasks, timeout, unit);
    }
    @Override
    public String toString() {
        return String.format("%s(%s)", beanName, executor);
    }
}
