package com.turbospaces.executor;

import java.time.Duration;
import java.util.Collections;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor.AbortPolicy;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.time.StopWatch;
import org.slf4j.MDC;
import org.springframework.beans.factory.BeanNameAware;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.common.PlatformUtil;

import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.binder.jvm.ExecutorServiceMetrics;
import io.vavr.CheckedFunction0;
import io.vavr.CheckedRunnable;
import lombok.experimental.Delegate;
import lombok.extern.slf4j.Slf4j;
import reactor.blockhound.integration.DefaultBlockHoundIntegration;

@Slf4j
public class DefaultPlatformExecutorService
        implements PlatformExecutorService, BeanNameAware, ExecutorService, InitializingBean, DisposableBean {
    private final ThreadLocal<Boolean> flag = DefaultBlockHoundIntegration.FLAG;
    private final ApplicationProperties props;
    private final MeterRegistry meterRegistry;
    private final BlockingQueue<Runnable> queue;
    private final RejectedExecutionHandler rejectPolicy;
    private String beanName;
    private int minPoolSize;
    private int maxPoolSize;

    @Delegate(types = { ExecutorService.class })
    private ThreadPoolExecutor executor;

    public DefaultPlatformExecutorService(ApplicationProperties props, MeterRegistry meterRegistry) {
        this(props, meterRegistry, 0, Math.max(Runtime.getRuntime().availableProcessors(), props.APP_DYNAMIC_PLATFORM_DEFAULT_MAX_SIZE.get()));
    }
    public DefaultPlatformExecutorService(ApplicationProperties props, MeterRegistry meterRegistry, int minPoolSize, int maxPoolSize) {
        this.props = Objects.requireNonNull(props);
        this.meterRegistry = Objects.requireNonNull(meterRegistry);
        this.minPoolSize = minPoolSize;
        this.maxPoolSize = maxPoolSize;
        this.queue = new SynchronousQueue<Runnable>();
        this.rejectPolicy = new AbortPolicy();
    }
    public DefaultPlatformExecutorService(ApplicationProperties props, MeterRegistry meterRegistry, int poolSize) {
        this(props, meterRegistry, poolSize, new LinkedBlockingQueue<Runnable>(), new LogQueueFullCallerRunsPolicy());
    }
    public DefaultPlatformExecutorService(
            ApplicationProperties props,
            MeterRegistry meterRegistry,
            int poolSize,
            BlockingQueue<Runnable> queue,
            RejectedExecutionHandler rejectPolicy) {
        this.props = Objects.requireNonNull(props);
        this.meterRegistry = Objects.requireNonNull(meterRegistry);
        this.minPoolSize = poolSize;
        this.maxPoolSize = poolSize;
        this.queue = Objects.requireNonNull(queue);
        this.rejectPolicy = Objects.requireNonNull(rejectPolicy);
    }
    @Override
    public void setBeanName(String name) {
        this.beanName = Objects.requireNonNull(name);
    }
    @Override
    public void afterPropertiesSet() {
        minPoolSize = props.cfg().getInteger(beanName + "." + "pool.min-size", minPoolSize);
        maxPoolSize = props.cfg().getInteger(beanName + "." + "pool.max-size", maxPoolSize);

        //
        // ~ thread factory
        //
        ThreadFactoryBuilder factory = new ThreadFactoryBuilder();
        factory.setDaemon(false);
        factory.setNameFormat(beanName + "-%d");
        factory.setThreadFactory(new ThreadFactory() {
            @Override
            public Thread newThread(Runnable r) {
                return new PlatformThread(props, r);
            }
        });

        //
        // ~ actual executor
        //
        executor = new ThreadPoolExecutor(
                minPoolSize,
                maxPoolSize,
                props.APP_PLATFORM_MAX_IDLE.get().toSeconds(),
                TimeUnit.SECONDS,
                queue,
                factory.build(),
                rejectPolicy);

        //
        // ~ bind metrics
        //
        ExecutorServiceMetrics metrics = new ExecutorServiceMetrics(executor, beanName, Collections.emptyList());
        metrics.bindTo(meterRegistry);
    }
    @Override
    public void destroy() {
        StopWatch stopWatch = StopWatch.createStarted();
        Duration timeout = props.APP_PLATFORM_GRACEFUL_SHUTDOWN_TIMEOUT.get();
        PlatformUtil.shutdownExecutor(executor, timeout);
        stopWatch.stop();

        log.debug("stopped executor-service: {} in {}", toString(), stopWatch);
    }
    @Override
    public void execute(Runnable command) {
        Map<String, String> mdcContextMap = MDC.getCopyOfContextMap();
        executor.execute(wrapRunnable(command, mdcContextMap));
    }
    @Override
    public ListenableFuture<?> submit(CheckedRunnable action) {
        //
        // ~ we can afford it, we know that even if we reach max threads, any new task will be rejected
        // ~ so we can safely assume that this is actually not blocking
        //
        boolean toReset = flag.get();
        try {
            flag.set(false);
            return PlatformExecutorService.super.submit(action);
        } finally {
            flag.set(toReset);
        }
    }
    @Override
    public <T> ListenableFuture<T> submit(CheckedFunction0<T> action) {
        //
        // ~ we can afford it, we know that even if we reach max threads, any new task will be rejected
        // ~ so we can safely assume that this is actually not blocking
        //
        boolean toReset = flag.get();
        try {
            flag.set(false);
            return PlatformExecutorService.super.submit(action);
        } finally {
            flag.set(toReset);
        }
    }
    @Override
    public String toString() {
        return String.format("%s(%s)", beanName, executor);
    }
}
