package com.turbospaces.aspects;

import java.lang.reflect.Method;
import java.util.Objects;

import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;

import com.turbospaces.annotations.NonBlockingCallOnlyGuard;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.common.NonBlockingCallOnly;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

@Aspect
@Slf4j
@RequiredArgsConstructor
public class NonBlockingCallOnlyAspect {
    private final ApplicationProperties props;

    @Around("@within(com.turbospaces.annotations.NonBlockingCallOnlyGuard)")
    public Object guardClass(ProceedingJoinPoint pjp) throws Throwable {
        Method method = ((MethodSignature) pjp.getSignature()).getMethod();
        Class<?> declaringClass = method.getDeclaringClass();
        NonBlockingCallOnlyGuard guard = declaringClass.getAnnotation(NonBlockingCallOnlyGuard.class);

        return perform(pjp, guard, method);
    }

    @Around("execution (@com.turbospaces.annotations.NonBlockingCallOnlyGuard * *.*(..))")
    public Object guardMethod(ProceedingJoinPoint pjp) throws Throwable {
        Method method = ((MethodSignature) pjp.getSignature()).getMethod();
        NonBlockingCallOnlyGuard guard = method.getAnnotation(NonBlockingCallOnlyGuard.class);
        if (Objects.isNull(guard)) {
            method = pjp.getTarget().getClass().getMethod(method.getName(), method.getParameterTypes());
            guard = method.getAnnotation(NonBlockingCallOnlyGuard.class);
        }

        return perform(pjp, guard, method);
    }

    private Object perform(ProceedingJoinPoint pjp, NonBlockingCallOnlyGuard guard, Method method) throws Throwable {
        if (Boolean.TRUE.equals(NonBlockingCallOnly.MARKER.get())) {
            var exception = new IllegalStateException(
                    String.format("api abuse detected: %s, blocking remote call in HTTP controller", pjp.toShortString())
            );
            log.error("api abuse detected", exception);
            if (props.isDevMode()) {
                throw exception;
            }
        }
        return pjp.proceed();
    }
}
