package com.turbospaces.properties;

import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import org.apache.commons.lang3.StringUtils;

import com.turbospaces.boot.AbstractBootstrapAware;

import lombok.NonNull;

public class BootstrapPropertiesParser extends AbstractBootstrapAware {

    private static final String SEPARATOR = ".";

    /**
     * Parse boot properties as a list of target type T.
     *
     * @param prefix the prefix to use when looking up properties.
     * @param mapper the mapper to use when converting properties to an object.
     *
     * @return the parsed properties.
    */
    public <T> List<T> parse(@NonNull String prefix, @NonNull BiFunction<String, Map<String, String>, T> mapper) {
        return keys().filter(key -> key.startsWith(prefix))
                .map(key -> extractName(prefix, key))
                .distinct()
                .map(name -> parse(prefix, mapper, name))
                .toList();
    }

    private <T> T parse(String prefix, BiFunction<String, Map<String, String>, T> mapper, String name) {
        String prefixName = String.join(SEPARATOR, prefix, name);
        var properties = keys().filter(key -> key.startsWith(prefixName))
                .collect(Collectors.toMap(key -> extractProperty(prefix, name, key), bootstrap.cfg()::getString));
        return mapper.apply(name, properties);
    }

    private Stream<String> keys() {
        return StreamSupport.stream(bootstrap.cfg().keys().spliterator(), false);
    }

    private String extractName(String prefix, String key) {
        return StringUtils.substringBetween(key, prefix + SEPARATOR, SEPARATOR);
    }

    private String extractProperty(String prefix, String name, String key) {
        return StringUtils.substringAfter(key, String.join(SEPARATOR, prefix, name) + SEPARATOR);
    }
}
