package com.turbospaces.rpc;

import java.time.Duration;
import java.time.Instant;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ScheduledFuture;
import java.util.function.Supplier;

import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.time.StopWatch;
import org.slf4j.MDC;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;

import com.google.common.base.Suppliers;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.SettableFuture;
import com.turbospaces.boot.Bootstrap;
import com.turbospaces.boot.BootstrapAware;
import com.turbospaces.executor.PlatformExecutorService;

import io.vavr.Function0;
import lombok.extern.slf4j.Slf4j;
import reactor.blockhound.integration.DefaultBlockHoundIntegration;

@SuppressWarnings("serial")
@Slf4j
public class CompletableRequestReplyMapper<K, V> extends ThreadPoolTaskScheduler implements BootstrapAware, RequestReplyMapper<K, V> {
    protected Supplier<Boolean> reportToSentryOnTimeout = Suppliers.ofInstance(true);
    protected ScheduledFuture<?> cleanUp;
    protected Duration timeout = Duration.ofMinutes(3);
    protected Cache<K, SettableFuture<V>> corr;

    public CompletableRequestReplyMapper() {
        super();
        setDaemon(true);
        setRemoveOnCancelPolicy(true);
        setPoolSize(Runtime.getRuntime().availableProcessors());
    }
    @Override
    public void setBootstrap(Bootstrap bootstrap) {
        timeout = bootstrap.props().BATCH_COMPLETION_TIMEOUT.get().plusMinutes(1);
    }
    @Override
    public void afterPropertiesSet() {
        super.afterPropertiesSet();

        //
        // ~ holder of all responses (god object)
        //
        corr = CacheBuilder.newBuilder().removalListener(new RemovalListener<K, SettableFuture<V>>() {
            @Override
            public void onRemoval(RemovalNotification<K, SettableFuture<V>> notification) {
                if (Objects.nonNull(notification.getCause())) {
                    String type = notification.getCause().name().toLowerCase().intern();
                    log.trace("onRemoval({}): {} {}", notification.getKey(), type, notification);
                }
            }
        }).expireAfterWrite(timeout).build();

        //
        // ~ useful reporting in logs
        //
        cleanUp = scheduleWithFixedDelay(new Runnable() {
            @Override
            public void run() {
                long size = corr.size();
                if (size > 0) {
                    log.debug("about to cleanUp correlation map of {} items ...", size);
                }

                corr.cleanUp();
            }
        }, Duration.ofMinutes(1));
    }
    @Override
    public void destroy() {
        super.destroy();
        if (Objects.nonNull(cleanUp)) {
            cleanUp.cancel(false);
        }
    }
    @Override
    public SettableFuture<V> acquire(K key, Duration duration) {
        Map<String, String> mdc = MDC.getCopyOfContextMap(); // ~ capture MDC
        StopWatch stopWatch = StopWatch.createStarted();

        SettableFuture<V> toReturn = SettableFuture.create();
        SettableFuture<V> putIfAbsent = putIfAbsent(key, toReturn);
        if (Objects.nonNull(putIfAbsent)) {
            toReturn.setException(new IllegalArgumentException("duplicate key violation for correlation id: " + key));
        } else {
            //
            // ~ timeout task - complete future with timeout exception just in case
            //
            ScheduledFuture<?> timerTask = schedule(new Runnable() {
                @Override
                public void run() {
                    SettableFuture<V> tmp = remove(key);
                    if (Objects.nonNull(tmp)) {
                        //
                        // ~ only complete when necessary
                        //
                        if (BooleanUtils.isFalse(tmp.isDone())) {
                            PlatformExecutorService.propagete(mdc);
                            try {
                                tmp.setException(new RequestReplyTimeout(duration, key, reportToSentryOnTimeout.get()));
                                log.info("request-reply(m={}) removed subj due to timeout", key);
                            } catch (Exception err) {
                                log.error(err.getMessage(), err);
                            } finally {
                                MDC.clear();
                            }
                        }
                    }
                }
            }, Instant.now().plus(duration));

            Futures.addCallback(toReturn, new FutureCallback<V>() {
                @Override
                public void onSuccess(V result) {
                    try {
                        //
                        // ~ cancelling the timer under the hood is blocking operation under some circumstances
                        //
                        DefaultBlockHoundIntegration.allowBlocking(new Runnable() {
                            @Override
                            public void run() {
                                timerTask.cancel(false);
                                stopWatch.stop();
                                log.debug("request-reply(m={}) completed in {}", key, stopWatch);
                            }
                        });
                    } finally {
                        remove(key);
                    }
                }
                @Override
                public void onFailure(Throwable t) {
                    log.atTrace().setCause(t).log();
                }
            }, MoreExecutors.directExecutor());
        }

        return toReturn;
    }
    @Override
    public boolean contains(K corrId) {
        return DefaultBlockHoundIntegration.allowBlockingUnchecked(new Function0<Boolean>() {
            @Override
            public Boolean apply() {
                return Objects.nonNull(corr.getIfPresent(corrId));
            }
        });
    }
    @Override
    public void complete(K key, V value) {
        Objects.requireNonNull(key);
        SettableFuture<V> subj = remove(key);
        if (Objects.nonNull(subj)) {
            setValue(subj, value);
        } else {
            log.trace("no such correlation for key: {}", key);
        }
    }
    @Override
    public void completeExceptionally(K key, Throwable reason) {
        Objects.requireNonNull(key);
        SettableFuture<V> subj = remove(key);
        if (Objects.nonNull(subj)) {
            setException(subj, reason);
        } else {
            log.trace("no such correlation for key: {}", key);
        }
    }
    @Override
    public void clear() {
        corr.invalidateAll();
    }
    @Override
    public int pendingCount() {
        return corr.asMap().size();
    }
    protected void setValue(SettableFuture<V> subj, V value) {
        subj.set(value);
    }
    protected void setException(SettableFuture<V> subj, Throwable reason) {
        subj.setException(reason);
    }
    private SettableFuture<V> putIfAbsent(K key, SettableFuture<V> toReturn) {
        ConcurrentMap<K, SettableFuture<V>> map = corr.asMap();
        return DefaultBlockHoundIntegration.allowBlockingUnchecked(new Function0<SettableFuture<V>>() {
            @Override
            public SettableFuture<V> apply() {
                return map.putIfAbsent(key, toReturn);
            }
        });
    }
    private SettableFuture<V> remove(K key) {
        return DefaultBlockHoundIntegration.allowBlockingUnchecked(new Function0<SettableFuture<V>>() {
            @Override
            public SettableFuture<V> apply() {
                return corr.asMap().remove(key);
            }
        });
    }
}
