package com.turbospaces.executor;

import java.time.Duration;
import java.util.Collections;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor.AbortPolicy;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.time.StopWatch;
import org.slf4j.MDC;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.turbospaces.boot.AbstractBootstrapAware;
import com.turbospaces.common.PlatformUtil;

import io.micrometer.core.instrument.binder.jvm.ExecutorServiceMetrics;
import io.vavr.CheckedFunction0;
import io.vavr.CheckedRunnable;
import lombok.experimental.Delegate;
import reactor.blockhound.integration.DefaultBlockHoundIntegration;

public class DefaultPlatformExecutorService
        extends AbstractBootstrapAware
        implements PlatformExecutorService, ExecutorService, InitializingBean, DisposableBean {
    private final ThreadLocal<Boolean> flag = DefaultBlockHoundIntegration.FLAG;
    private final String name;
    private final int minPoolSize;
    private final int maxPoolSize;
    @Delegate
    private ExecutorService executor;

    public DefaultPlatformExecutorService(String name, int minPoolSize, int maxPoolSize) {
        this.name = Objects.requireNonNull(name);
        this.minPoolSize = minPoolSize;
        this.maxPoolSize = maxPoolSize;
    }
    @Override
    public void afterPropertiesSet() {
        ThreadFactoryBuilder factory = new ThreadFactoryBuilder();
        factory.setDaemon(false);
        factory.setNameFormat(name + "-%d");
        factory.setThreadFactory(new ThreadFactory() {
            @Override
            public Thread newThread(Runnable r) {
                PlatformThread thread = new PlatformThread(r);
                thread.setBootstrap(bootstrap);
                return thread;
            }
        });

        Duration ttl = bootstrap.props().APP_PLATFORM_MAX_IDLE.get();
        executor = new ThreadPoolExecutor(
                minPoolSize,
                maxPoolSize,
                ttl.toSeconds(),
                TimeUnit.SECONDS,
                new SynchronousQueue<Runnable>(),
                factory.build(),
                new AbortPolicy());

        ExecutorServiceMetrics metrics = new ExecutorServiceMetrics(executor, name, Collections.emptyList());
        metrics.bindTo(bootstrap.meterRegistry());
    }
    @Override
    public void destroy() {
        StopWatch stopWatch = StopWatch.createStarted();
        Duration timeout = bootstrap.props().APP_PLATFORM_GRACEFUL_SHUTDOWN_TIMEOUT.get();
        PlatformUtil.shutdownExecutor(executor, timeout);
        stopWatch.stop();

        logger.debug("stopped executor-service: {} in {}", executor, stopWatch);
    }
    @Override
    public void execute(Runnable command) {
        Map<String, String> mdcContextMap = MDC.getCopyOfContextMap();
        executor.execute(wrapRunnable(command, mdcContextMap));
    }
    @Override
    public ListenableFuture<?> submit(CheckedRunnable action) {
        //
        // ~ we can afford it, we know that even if we reach max threads, any new task will be rejected
        // ~ so we can safely assume that this is actually not blocking
        //
        boolean toReset = flag.get();
        try {
            flag.set(false);
            return PlatformExecutorService.super.submit(action);
        } finally {
            flag.set(toReset);
        }
    }
    @Override
    public <T> ListenableFuture<T> submit(CheckedFunction0<T> action) {
        //
        // ~ we can afford it, we know that even if we reach max threads, any new task will be rejected
        // ~ so we can safely assume that this is actually not blocking
        //
        boolean toReset = flag.get();
        try {
            flag.set(false);
            return PlatformExecutorService.super.submit(action);
        } finally {
            flag.set(toReset);
        }
    }
}
