package com.turbospaces.boot;

import java.io.InputStream;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.time.LocalDate;
import java.time.ZoneOffset;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;

import org.apache.commons.lang3.RandomStringUtils;
import org.eclipse.jgit.lib.Repository;
import org.springframework.cloud.SmartCloud;
import org.springframework.cloud.service.ServiceInfo;
import org.springframework.cloud.service.UriBasedServiceInfo;
import org.springframework.util.ResourceUtils;

import com.google.common.collect.Maps;
import com.google.common.net.HostAndPort;
import com.netflix.archaius.api.exceptions.ConfigException;
import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.CloudOptions;
import com.turbospaces.common.PlatformUtil;
import com.turbospaces.ups.PlainServiceInfo;
import com.turbospaces.ups.RawServiceInfo;
import com.turbospaces.ups.UPSs;

import reactor.core.Disposable;

public final class MockCloud implements SmartCloud, Disposable {
    private final Properties cloudPropsOverride = new Properties();
    private final ApplicationConfig cfg;
    private final Map<String, ServiceInfo> services = Maps.newHashMap();
    private final String spaceName;
    private final String appId;
    private final String appName;

    private MockCloud(String appId, String appName) throws ConfigException {
        this.appId = Objects.requireNonNull(appId);
        this.appName = Objects.requireNonNull(appName);
        this.spaceName = RandomStringUtils.randomAlphabetic(4);
        this.cfg = ApplicationConfig.create();
    }
    @Override
    public void addUps(ServiceInfo info) {
        services.put(info.getId(), info);
    }
    @Override
    public boolean removeUps(String id) {
        return Objects.nonNull(services.remove(id));
    }
    @Override
    public void dispose() {
        services.clear();
    }
    public MockCloud witCfg(int port, Repository repo) {
        return withCfg(HostAndPort.fromParts("localhost", port), repo);
    }
    public MockCloud withCfg(HostAndPort addr, Repository repo) {
        PlainServiceInfo info = new PlainServiceInfo(UPSs.CFG, PlainServiceInfo.HTTP_SCHEME, addr, null, null, repo.getIdentifier());
        addUps(info);
        return this;
    }
    public MockCloud appPort(int port) {
        cloudPropsOverride.put(CloudOptions.CLOUD_APP_PORT, port);
        return this;
    }
    public ApplicationConfig build() throws Exception {
        String slot = "0";
        int port = PlatformUtil.findAvailableTcpPort();
        String hostname = PlatformUtil.detectIp();

        cfg.setLocalProperty("kafka.auto.offset.reset", "earliest"); // consume from the beginning

        //
        // ~ mock typical cloud properties
        //
        Properties props = new Properties();
        props.put(CloudOptions.CLOUD_APP_ID, appId);
        props.put(CloudOptions.CLOUD_APP_SPACE_NAME, spaceName);
        props.put(CloudOptions.CLOUD_APP_HOST, hostname);
        props.put(CloudOptions.CLOUD_APP_PORT, port);
        props.put(CloudOptions.CLOUD_APP_INSTANCE_INDEX, slot);
        props.put(CloudOptions.CLOUD_APP_NAME, appName);
        props.putAll(cloudPropsOverride);

        //
        // ~ put services
        //
        for (ServiceInfo info : services.values()) {
            String uri = null;
            if (info instanceof UriBasedServiceInfo) {
                uri = ((UriBasedServiceInfo) info).getUri();
            } else if (info instanceof RawServiceInfo raw) {
                uri = raw.toByteSource().asCharSource(StandardCharsets.UTF_8).read();
            } else {
                throw new IllegalArgumentException();
            }

            props.put("service." + info.getId() + ".uri", uri);
        }

        cfg.setLocalProperties(props);

        return cfg;
    }
    public static MockCloud fromClasspathAppProps() throws Exception {
        //
        // ~ load required properties file
        //
        URL resource = ResourceUtils.getURL(ResourceUtils.CLASSPATH_URL_PREFIX + "application.properties");
        return fromClasspathAppProps(resource);
    }
    public static MockCloud fromClasspathAppProps(URL url) throws Exception {
        //
        // ~ load required properties file
        //
        try (InputStream io = url.openStream()) {
            Properties props = new Properties();
            props.load(io);
            String appId = props.getProperty(CloudOptions.CLOUD_APP_ID);
            String appName = props.getProperty(CloudOptions.CLOUD_APP_NAME, randomAppName());
            return new MockCloud(appId, appName);
        }
    }
    public static MockCloud newMock() throws ConfigException {
        int length = MockCloud.class.getSimpleName().length();
        String randomAppId = "app-" + RandomStringUtils.randomAlphanumeric(length);
        return new MockCloud(randomAppId, randomAppName());
    }
    private static String randomAppName() {
        LocalDate now = LocalDate.now(ZoneOffset.UTC);
        String randomGroup = "group-" + RandomStringUtils.randomAlphanumeric(2);
        String randomAppId = "app-" + RandomStringUtils.randomAlphanumeric(4);
        String randomVersion = now.getYear() + "" + now.getMonthValue() + "" + now.getDayOfMonth();

        return String.format("%s:%s:%s", randomGroup, randomAppId, randomVersion);
    }
}
