package com.turbospaces.rpc;

import java.time.Duration;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;

import com.google.common.util.concurrent.FluentFuture;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import com.google.protobuf.Any;
import com.google.protobuf.Timestamp;
import com.turbospaces.api.facade.DefaultRequestWrapperFacade;
import com.turbospaces.api.facade.MockResponseWrapperFacade;
import com.turbospaces.api.facade.ResponseWrapperFacade;
import com.turbospaces.boot.MockCloud;
import com.turbospaces.boot.SimpleBootstrap;
import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.dispatch.AbstractSafeResponseConsumer;

import api.v1.ApiFactory;
import api.v1.MockApiFactory;
import api.v1.MockResponseWrapper;
import api.v1.RequestWrapper;
import lombok.extern.slf4j.Slf4j;

@Slf4j
class SafeResponseConsumerTest {
    @Test
    void works() throws Throwable {
        ApplicationConfig cfg = MockCloud.newMock().build();
        ApplicationProperties props = new ApplicationProperties(cfg);
        SimpleBootstrap boot = new SimpleBootstrap(props);
        boot.run();

        try {
            SettableFuture<ResponseWrapperFacade> future = SettableFuture.create();

            var req = Timestamp.newBuilder().setSeconds(TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis())).build();
            var headers = api.v1.Headers.newBuilder().setMessageId(UUID.randomUUID().toString()).build();
            var apiFactory = new MockApiFactory();
            var reqw = new DefaultRequestWrapperFacade(
                    apiFactory.eventTemplate(),
                    RequestWrapper.newBuilder()
                            .setHeaders(headers)
                            .setBody(Any.pack(req)));

            boot.globalPlatform().execute(new Runnable() {
                @Override
                public void run() {
                    future.set(new MockResponseWrapperFacade(apiFactory.eventTemplate(), new MockResponseWrapper(headers, Any.pack(req))));
                }
            });

            var apiResponse = new DefaultApiResponse<>(future, apiFactory, Timestamp.class, reqw);
            CountDownLatch latch = new CountDownLatch(1);
            AbstractSafeResponseConsumer consumer = new AbstractSafeResponseConsumer(apiResponse, Mockito.mock(ApiFactory.class)) {
                @Override
                public void accept(ResponseWrapperFacade t) throws Throwable {
                    latch.countDown();
                }
            };

            consumer.run();
            latch.await();
        } finally {
            boot.shutdown();
        }
    }
    @Test
    void exception() throws Throwable {
        ApplicationConfig cfg = MockCloud.newMock().build();
        ApplicationProperties props = new ApplicationProperties(cfg);
        SimpleBootstrap boot = new SimpleBootstrap(props);
        boot.run();

        try {
            SettableFuture<ResponseWrapperFacade> future = SettableFuture.create();

            var req = Timestamp.newBuilder().setSeconds(TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis())).build();
            var headers = api.v1.Headers.newBuilder().setMessageId(UUID.randomUUID().toString()).build();
            var apiFactory = new MockApiFactory();
            var reqw = new DefaultRequestWrapperFacade(
                    apiFactory.eventTemplate(),
                    RequestWrapper.newBuilder()
                            .setHeaders(headers)
                            .setBody(Any.pack(req)));

            boot.globalPlatform().execute(new Runnable() {
                @Override
                public void run() {
                    future.set(new MockResponseWrapperFacade(apiFactory.eventTemplate(), new MockResponseWrapper(headers, Any.pack(req))));
                }
            });

            var apiResponse = new DefaultApiResponse<>(future, apiFactory, Timestamp.class, reqw);
            CountDownLatch latch = new CountDownLatch(1);

            AbstractSafeResponseConsumer consumer = new AbstractSafeResponseConsumer(apiResponse, Mockito.mock(ApiFactory.class)) {
                @Override
                public void onFailure(Throwable t) {
                    super.onFailure(t);
                    latch.countDown();
                }
                @Override
                public void accept(ResponseWrapperFacade t) throws Throwable {
                    throw new IllegalArgumentException();
                }
            };

            consumer.run();
            latch.await();
        } finally {
            boot.shutdown();
        }
    }
    @Test
    void timeout() throws Throwable {
        ApplicationConfig cfg = MockCloud.newMock().build();
        ApplicationProperties props = new ApplicationProperties(cfg);
        SimpleBootstrap boot = new SimpleBootstrap(props);
        boot.run();

        try {
            ListenableFuture<ResponseWrapperFacade> future = SettableFuture.create();
            ListenableFuture<ResponseWrapperFacade> fluent = FluentFuture.from(future).withTimeout(Duration.ofMillis(1), boot.globalPlatform());
            var req = Timestamp.newBuilder().setSeconds(TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis())).build();
            var apiFactory = new MockApiFactory();
            var reqw = new DefaultRequestWrapperFacade(
                    apiFactory.eventTemplate(),
                    RequestWrapper.newBuilder()
                            .setHeaders(api.v1.Headers.newBuilder().setMessageId(UUID.randomUUID().toString()))
                            .setBody(Any.pack(req)));

            var apiResponse = new DefaultApiResponse<>(fluent, apiFactory, Timestamp.class, reqw);
            CountDownLatch latch = new CountDownLatch(1);
            AtomicReference<Boolean> code = new AtomicReference<>();
            AbstractSafeResponseConsumer consumer = new AbstractSafeResponseConsumer(apiResponse, apiFactory) {
                @Override
                public void accept(ResponseWrapperFacade t) throws Throwable {
                    code.set(t.status().isTimeout());
                    log.debug(t.status().errorText());
                    latch.countDown();
                }
            };

            consumer.run();
            latch.await();

            Assertions.assertTrue(code.get());
        } finally {
            boot.shutdown();
        }
    }
    @Test
    void timeoutWithMapper() throws Throwable {
        ApplicationConfig cfg = MockCloud.newMock().build();
        ApplicationProperties props = new ApplicationProperties(cfg);
        SimpleBootstrap boot = new SimpleBootstrap(props);
        boot.run();

        try {
            CompletableRequestReplyMapper<String, ResponseWrapperFacade> mapper = new CompletableRequestReplyMapper<>();
            mapper.afterPropertiesSet();

            ListenableFuture<ResponseWrapperFacade> future = mapper.acquire(UUID.randomUUID().toString(), Duration.ofMillis(1));

            var req = Timestamp.newBuilder().setSeconds(TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis())).build();
            var apiFactory = new MockApiFactory();
            var reqw = new DefaultRequestWrapperFacade(
                    apiFactory.eventTemplate(),
                    RequestWrapper.newBuilder()
                            .setHeaders(api.v1.Headers.newBuilder().setMessageId(UUID.randomUUID().toString()))
                            .setBody(Any.pack(req)));

            var apiResponse = new DefaultApiResponse<>(future, apiFactory, Timestamp.class, reqw);
            CountDownLatch latch = new CountDownLatch(1);
            AtomicReference<Boolean> code = new AtomicReference<>();
            AbstractSafeResponseConsumer consumer = new AbstractSafeResponseConsumer(apiResponse, apiFactory) {
                @Override
                public void accept(ResponseWrapperFacade t) throws Throwable {
                    code.set(t.status().isTimeout());
                    log.debug(t.status().errorText());
                    latch.countDown();
                }
            };

            consumer.run();
            latch.await();
            mapper.destroy();

            Assertions.assertTrue(code.get());
        } finally {
            boot.shutdown();
        }
    }
}
