package com.turbospaces.executor;

import java.time.Duration;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.time.StopWatch;
import org.slf4j.MDC;
import org.springframework.beans.factory.DisposableBean;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.cache.RemovalCause;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import com.turbospaces.boot.AbstractBootstrapAware;
import com.turbospaces.boot.Bootstrap;
import com.turbospaces.common.PlatformUtil;

import io.netty.util.AsciiString;

public class ThreadPoolContextWorker extends AbstractBootstrapAware implements ContextWorker, DisposableBean {
    private static EnumSet<RemovalCause> REASONS = EnumSet.of(RemovalCause.EXPIRED, RemovalCause.SIZE);
    private final ExecutorService executor;
    private LoadingCache<AsciiString, SerialContextWorker> executors;

    public ThreadPoolContextWorker(ExecutorService executor) {
        this.executor = Objects.requireNonNull(executor);
    }
    @Override
    public void setBootstrap(Bootstrap bootstrap) throws Throwable {
        super.setBootstrap(bootstrap);

        Duration timeout = bootstrap.props().BATCH_COMPLETION_TIMEOUT.get().plusMinutes(1);
        Duration cleanupInterval = bootstrap.props().APP_TIMER_INTERVAL.get();

        //
        // ~ we should try to free occupied memory but keep for reasonable time
        //
        executors = CacheBuilder.newBuilder()
                .expireAfterAccess(timeout.toMinutes(), TimeUnit.MINUTES)
                .removalListener(new RemovalListener<Object, Object>() {
                    @Override
                    public void onRemoval(RemovalNotification<Object, Object> notification) {
                        if (Objects.nonNull(notification.getCause())) {
                            if (REASONS.contains(notification.getCause())) {
                                String type = notification.getCause().name().toLowerCase().intern();
                                logger.trace("onRemoval({}): {}", type, notification.getKey());
                            }
                        }
                    }
                })
                .build(new CacheLoader<AsciiString, SerialContextWorker>() {
                    @Override
                    public SerialContextWorker load(AsciiString key) {
                        return new SerialContextWorker(key, executor, bootstrap.meterRegistry());
                    }
                });

        //
        // ~ perform corresponding maintenance (crucial for memory cleanUp)
        //
        bootstrap.globalPlatform().scheduleAtFixedRate(new Runnable() {
            @Override
            public void run() {
                long size = executors.size();
                if (size > 0) {
                    logger.debug("about to cleanUp executors cache of {} items ...", size);
                }
                executors.cleanUp();
            }
        }, cleanupInterval.toSeconds(), cleanupInterval.toSeconds(), TimeUnit.SECONDS);
    }
    @Override
    public ContextWorker forKey(WorkUnit unit) {
        AsciiString partitionKey = new AsciiString(unit.key());
        return executors.getUnchecked(partitionKey);
    }
    @Override
    public void execute(Runnable command) {
        Map<String, String> mdc = MDC.getCopyOfContextMap();
        executor.execute(wrapRunnable(command, mdc));
    }
    @Override
    public void destroy() throws Exception {
        StopWatch stopWatch = StopWatch.createStarted();
        Duration timeout = bootstrap.props().APP_PLATFORM_GRACEFUL_SHUTDOWN_TIMEOUT.get();
        List<Runnable> never = PlatformUtil.shutdownExecutor(executor, timeout);
        stopWatch.stop();
        logger.debug("stopped worker executor in {}. {} task(s) were never executed ...", stopWatch, never.size());
    }
}
