package com.turbospaces.rpc;

import java.util.UUID;
import java.util.function.BooleanSupplier;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.MDC;

import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.SettableFuture;
import com.turbospaces.api.facade.ResponseWrapperFacade;
import com.turbospaces.boot.Bootstrap;
import com.turbospaces.mdc.MdcTags;

import io.micrometer.core.instrument.ImmutableTag;
import io.micrometer.core.instrument.Tag;

@SuppressWarnings("serial")
public class DefaultRequestReplyMapper extends CompletableRequestReplyMapper<UUID, ResponseWrapperFacade> {
    private Bootstrap bootstrap;

    @Override
    public void setBootstrap(Bootstrap bootstrap) {
        super.setBootstrap(bootstrap);

        this.bootstrap = bootstrap;
        this.reportToSentryOnTimeout = bootstrap.props().SENTRY_REQUEST_REPLY_TIMEOUT_REPORT;
    }
    @Override
    protected void setValue(SettableFuture<ResponseWrapperFacade> subj, ResponseWrapperFacade value) {
        ImmutableList.Builder<Tag> tags = ImmutableList.builder();
        addMetricFromMdcIfPresent(MdcTags.MDC_OPERATION, tags);
        addMetricFromMdcIfPresent(MdcTags.MDC_TOPIC, tags);
        tags.add(new ImmutableTag(MdcTags.MDC_ERROR_CLASS, "none"));

        var timer = bootstrap.meterRegistry().timer("reply.complete.latency", tags.build());
        timer.record(new BooleanSupplier() {
            @Override
            public boolean getAsBoolean() {
                return subj.set(value);
            }
        });
    }
    @Override
    protected void setException(SettableFuture<ResponseWrapperFacade> subj, Throwable reason) {
        ImmutableList.Builder<Tag> tags = ImmutableList.builder();
        addMetricFromMdcIfPresent(MdcTags.MDC_OPERATION, tags);
        addMetricFromMdcIfPresent(MdcTags.MDC_TOPIC, tags);
        tags.add(new ImmutableTag(MdcTags.MDC_ERROR_CLASS, reason.getClass().getSimpleName()));

        var timer = bootstrap.meterRegistry().timer("reply.complete.latency", tags.build());
        timer.record(new BooleanSupplier() {
            @Override
            public boolean getAsBoolean() {
                return subj.setException(reason);
            }
        });
    }
    private static void addMetricFromMdcIfPresent(String key, ImmutableList.Builder<Tag> tags) {
        String mdc = MDC.get(key);
        if (StringUtils.isNotEmpty(mdc)) {
            tags.add(new ImmutableTag(key, mdc));
        }
    }
}
