package com.turbospaces.rpc;

import java.util.Objects;

import javax.annotation.OverridingMethodsMustInvokeSuper;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.util.concurrent.FutureCallback;
import com.turbospaces.boot.Bootstrap;
import com.turbospaces.boot.BootstrapAware;

public class FutureCallbackWrapper<T> implements BootstrapAware, FutureCallback<T> {
    private final FutureCallback<T> callback;
    private final Thread thread;
    private boolean devMode = true;

    public FutureCallbackWrapper(FutureCallback<T> callback) {
        this.callback = Objects.requireNonNull(callback);
        this.thread = Thread.currentThread();
    }
    @Override
    @OverridingMethodsMustInvokeSuper
    public void setBootstrap(Bootstrap bootstrap) {
        devMode = bootstrap.isDevMode();
    }
    @Override
    public void onSuccess(T result) {
        runGuard();
        callback.onSuccess(result);
    }
    @Override
    public void onFailure(Throwable t) {
        runGuard();
        callback.onFailure(t);
    }
    private void runGuard() {
        Thread currentThread = Thread.currentThread();
        String name = callback.getClass().getSimpleName();

        //
        // ~ we should not execute callback in same thread as it was submitted
        //
        if (thread.equals(currentThread)) {
            if (devMode) {
                var reason = "detected callback execution abuse in: " + name;
                var err = new IllegalStateException(reason);
                err.fillInStackTrace();

                throw err;
            }

            Logger log = LoggerFactory.getLogger(name);
            log.error("detected callback execution of {} in origin: {}", name, thread);
        }
    }
}
