package com.turbospaces.rpc;

import java.io.IOException;
import java.io.OutputStream;
import java.time.Duration;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import org.mockito.ArgumentMatchers;
import org.mockito.Mockito;

import com.google.common.util.concurrent.FluentFuture;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import com.google.protobuf.Any;
import com.google.protobuf.ByteString;
import com.google.protobuf.Timestamp;
import com.turbospaces.api.facade.HeadersFacade;
import com.turbospaces.api.facade.RequestWrapperFacade;
import com.turbospaces.api.facade.ResponseStatusFacade;
import com.turbospaces.api.facade.ResponseWrapperFacade;
import com.turbospaces.api.mappers.ResponseFacadeMapper;
import com.turbospaces.boot.MockCloud;
import com.turbospaces.boot.SimpleBootstrap;
import com.turbospaces.cfg.ApplicationConfig;
import com.turbospaces.cfg.ApplicationProperties;
import com.turbospaces.dispatch.AbstractSafeResponseConsumer;

import api.v1.ApiFactory;
import api.v1.CacheControl;
import lombok.extern.slf4j.Slf4j;

@Slf4j
class SafeResponseConsumerTest {
    @Test
    void works() throws Throwable {
        ApplicationConfig cfg = MockCloud.newMock().build();
        ApplicationProperties props = new ApplicationProperties(cfg);
        SimpleBootstrap boot = new SimpleBootstrap(props);
        boot.run();

        try {
            SettableFuture<ResponseWrapperFacade> future = SettableFuture.create();

            var req = Timestamp.newBuilder().setSeconds(TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis())).build();

            var respw = Mockito.mock(ResponseWrapperFacade.class);

            var reqw = Mockito.mock(RequestWrapperFacade.class);
            Mockito.when(reqw.body()).thenReturn(Any.pack(req));
            Mockito.when(reqw.toReply(ArgumentMatchers.any(), ArgumentMatchers.any(ResponseStatusFacade.class))).thenReturn(respw);

            boot.globalPlatform().execute(new Runnable() {
                @Override
                public void run() {
                    future.set(new ResponseWrapperFacade() {
                        @Override
                        public byte[] toBytes() {
                            return body().toByteArray();
                        }
                        @Override
                        public ByteString toByteString() {
                            return body().toByteString();
                        }
                        @Override
                        public void writeTo(OutputStream output) throws IOException {
                            body().writeTo(output);
                        }
                        @Override
                        public ResponseStatusFacade status() {
                            return Mockito.mock(ResponseStatusFacade.class);
                        }
                        @Override
                        public HeadersFacade headers() {
                            return Mockito.mock(HeadersFacade.class);
                        }
                        @Override
                        public int getSerializedSize() {
                            return body().getSerializedSize();
                        }
                        @Override
                        public Any body() {
                            return Any.pack(req);
                        }
                        @Override
                        public CacheControl cacheControl() {
                            return CacheControl.getDefaultInstance();
                        }
                    });
                }
            });

            var apiResponse = new DefaultApiResponse<>(future, Timestamp.class, reqw);
            CountDownLatch latch = new CountDownLatch(1);
            AbstractSafeResponseConsumer consumer = new AbstractSafeResponseConsumer(apiResponse, Mockito.mock(ApiFactory.class)) {
                @Override
                public void accept(ResponseWrapperFacade t) throws Throwable {
                    latch.countDown();
                }
            };

            consumer.run();
            latch.await();
        } finally {
            boot.shutdown();
        }
    }
    @Test
    void exception() throws Throwable {
        ApplicationConfig cfg = MockCloud.newMock().build();
        ApplicationProperties props = new ApplicationProperties(cfg);
        SimpleBootstrap boot = new SimpleBootstrap(props);
        boot.run();

        try {
            SettableFuture<ResponseWrapperFacade> future = SettableFuture.create();

            var req = Timestamp.newBuilder().setSeconds(TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis())).build();

            var respw = Mockito.mock(ResponseWrapperFacade.class);

            var reqw = Mockito.mock(RequestWrapperFacade.class);
            Mockito.when(reqw.body()).thenReturn(Any.pack(req));
            Mockito.when(reqw.toReply(ArgumentMatchers.any(), ArgumentMatchers.any(ResponseStatusFacade.class))).thenReturn(respw);

            boot.globalPlatform().execute(new Runnable() {
                @Override
                public void run() {
                    future.set(new ResponseWrapperFacade() {
                        @Override
                        public byte[] toBytes() {
                            return body().toByteArray();
                        }
                        @Override
                        public void writeTo(OutputStream output) throws IOException {
                            body().writeTo(output);
                        }
                        @Override
                        public ByteString toByteString() {
                            return body().toByteString();
                        }
                        @Override
                        public ResponseStatusFacade status() {
                            return Mockito.mock(ResponseStatusFacade.class);
                        }
                        @Override
                        public HeadersFacade headers() {
                            return Mockito.mock(HeadersFacade.class);
                        }
                        @Override
                        public int getSerializedSize() {
                            return body().getSerializedSize();
                        }
                        @Override
                        public Any body() {
                            return Any.pack(req);
                        }
                        @Override
                        public CacheControl cacheControl() {
                            return CacheControl.getDefaultInstance();
                        }
                    });
                }
            });

            var apiResponse = new DefaultApiResponse<>(future, Timestamp.class, reqw);
            CountDownLatch latch = new CountDownLatch(1);

            AbstractSafeResponseConsumer consumer = new AbstractSafeResponseConsumer(apiResponse, Mockito.mock(ApiFactory.class)) {
                @Override
                public void onFailure(Throwable t) {
                    super.onFailure(t);
                    latch.countDown();
                }
                @Override
                public void accept(ResponseWrapperFacade t) throws Throwable {
                    throw new IllegalArgumentException();
                }
            };

            consumer.run();
            latch.await();
        } finally {
            boot.shutdown();
        }
    }
    @Test
    void timeout() throws Throwable {
        ApplicationConfig cfg = MockCloud.newMock().build();
        ApplicationProperties props = new ApplicationProperties(cfg);
        SimpleBootstrap boot = new SimpleBootstrap(props);
        boot.run();

        try {
            ListenableFuture<ResponseWrapperFacade> future = SettableFuture.create();
            ListenableFuture<ResponseWrapperFacade> fluent = FluentFuture.from(future).withTimeout(Duration.ofMillis(1), boot.globalPlatform());
            var req = Timestamp.newBuilder().setSeconds(TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis())).build();

            var statusw = Mockito.mock(ResponseStatusFacade.class);
            Mockito.when(statusw.isTimeout()).thenReturn(true);

            var respw = Mockito.mock(ResponseWrapperFacade.class);
            Mockito.when(respw.status()).thenReturn(statusw);

            var respMapper = Mockito.mock(ResponseFacadeMapper.class);
            Mockito.when(respMapper.toTimeoutReplyWithoutBody(ArgumentMatchers.any())).thenReturn(respw);

            var apiFactory = Mockito.mock(ApiFactory.class);
            Mockito.when(apiFactory.responseMapper()).thenReturn(respMapper);

            var reqw = Mockito.mock(RequestWrapperFacade.class);
            Mockito.when(reqw.body()).thenReturn(Any.pack(req));

            var apiResponse = new DefaultApiResponse<>(fluent, Timestamp.class, reqw);
            CountDownLatch latch = new CountDownLatch(1);
            AtomicReference<Boolean> code = new AtomicReference<>();
            AbstractSafeResponseConsumer consumer = new AbstractSafeResponseConsumer(apiResponse, apiFactory) {
                @Override
                public void accept(ResponseWrapperFacade t) throws Throwable {
                    code.set(t.status().isTimeout());
                    log.debug(t.status().errorText());
                    latch.countDown();
                }
            };

            consumer.run();
            latch.await();

            Assertions.assertTrue(code.get());
        } finally {
            boot.shutdown();
        }
    }
    @Test
    void timeoutWithMapper() throws Throwable {
        ApplicationConfig cfg = MockCloud.newMock().build();
        ApplicationProperties props = new ApplicationProperties(cfg);
        SimpleBootstrap boot = new SimpleBootstrap(props);
        boot.run();

        try {
            CompletableRequestReplyMapper<String, ResponseWrapperFacade> mapper = new CompletableRequestReplyMapper<>();
            mapper.initialize();
            ListenableFuture<ResponseWrapperFacade> future = mapper.acquire(UUID.randomUUID().toString(), Duration.ofMillis(1));

            var req = Timestamp.newBuilder().setSeconds(TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis())).build();

            var statusw = Mockito.mock(ResponseStatusFacade.class);
            Mockito.when(statusw.isTimeout()).thenReturn(true);

            var respw = Mockito.mock(ResponseWrapperFacade.class);
            Mockito.when(respw.status()).thenReturn(statusw);

            var respMapper = Mockito.mock(ResponseFacadeMapper.class);
            Mockito.when(respMapper.toTimeoutReplyWithoutBody(ArgumentMatchers.any())).thenReturn(respw);

            var apiFactory = Mockito.mock(ApiFactory.class);
            Mockito.when(apiFactory.responseMapper()).thenReturn(respMapper);

            var reqw = Mockito.mock(RequestWrapperFacade.class);
            Mockito.when(reqw.body()).thenReturn(Any.pack(req));

            var apiResponse = new DefaultApiResponse<>(future, Timestamp.class, reqw);
            CountDownLatch latch = new CountDownLatch(1);
            AtomicReference<Boolean> code = new AtomicReference<>();
            AbstractSafeResponseConsumer consumer = new AbstractSafeResponseConsumer(apiResponse, apiFactory) {
                @Override
                public void accept(ResponseWrapperFacade t) throws Throwable {
                    code.set(t.status().isTimeout());
                    log.debug(t.status().errorText());
                    latch.countDown();
                }
            };

            consumer.run();
            latch.await();
            mapper.destroy();

            Assertions.assertTrue(code.get());
        } finally {
            boot.shutdown();
        }
    }
}
