package com.turbospaces.executor;

import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.time.StopWatch;
import org.slf4j.MDC;
import org.springframework.beans.factory.DisposableBean;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.turbospaces.boot.AbstractBootstrapAware;
import com.turbospaces.common.PlatformUtil;

import io.netty.util.AsciiString;

public class ThreadPoolContextWorker extends AbstractBootstrapAware implements ContextWorker, DisposableBean {
    private final LoadingCache<AsciiString, SerialContextWorker> executors;
    private final ExecutorService executor;

    public ThreadPoolContextWorker(ExecutorService executor) {
        this.executor = Objects.requireNonNull(executor);
        this.executors = CacheBuilder.newBuilder()
                .expireAfterAccess(1, TimeUnit.HOURS)
                .build(new CacheLoader<AsciiString, SerialContextWorker>() {
                    @Override
                    public SerialContextWorker load(AsciiString key) {
                        return new SerialContextWorker(key, executor);
                    }
                });
    }
    @Override
    public ContextWorker forKey(WorkUnit unit) {
        AsciiString partitionKey = new AsciiString(unit.key());
        return executors.getUnchecked(partitionKey);
    }
    @Override
    public void execute(Runnable command) {
        Map<String, String> mdcContextMap = MDC.getCopyOfContextMap();
        executor.execute(wrapRunnable(command, mdcContextMap));
    }
    @Override
    public void destroy() throws Exception {
        StopWatch stopWatch = StopWatch.createStarted();
        Duration timeout = bootstrap.props().APP_PLATFORM_GRACEFUL_SHUTDOWN_TIMEOUT.get();
        List<Runnable> never = PlatformUtil.shutdownExecutor(executor, timeout);
        stopWatch.stop();
        logger.debug("stopped kafka-worker in {}. {} task(s) were never executed ...", stopWatch, never.size());
    }
}
